/*
 *
 *  Copyright French Prime minister Office/SGMAP/DINSIC/Vitam Program (2019-2022)
 *  and the signatories of the "VITAM - Accord du Contributeur" agreement.
 *
 *  contact@programmevitam.fr
 *
 *  This software is a computer program whose purpose is to implement
 *  implement a digital archiving front-office system for the secure and
 *  efficient high volumetry VITAM solution.
 *
 *  This software is governed by the CeCILL-C license under French law and
 *  abiding by the rules of distribution of free software.  You can  use,
 *  modify and/ or redistribute the software under the terms of the CeCILL-C
 *  license as circulated by CEA, CNRS and INRIA at the following URL
 *  "http://www.cecill.info".
 *
 *  As a counterpart to the access to the source code and  rights to copy,
 *  modify and redistribute granted by the license, users are provided only
 *  with a limited warranty  and the software's author,  the holder of the
 *  economic rights,  and the successive licensors  have only  limited
 *  liability.
 *
 *  In this respect, the user's attention is drawn to the risks associated
 *  with loading,  using,  modifying and/or developing or reproducing the
 *  software by the user in light of its specific status of free software,
 *  that may mean  that it is complicated to manipulate,  and  that  also
 *  therefore means  that it is reserved for developers  and  experienced
 *  professionals having in-depth computer knowledge. Users are therefore
 *  encouraged to load and test the software's suitability as regards their
 *  requirements in conditions enabling the security of their systems and/or
 *  data to be ensured and,  more generally, to use and operate it in the
 *  same conditions as regards security.
 *
 *  The fact that you are presently reading this means that you have had
 *  knowledge of the CeCILL-C license and that you accept its terms.
 *
 */

package fr.gouv.vitamui.archives.search.server.service;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.opencsv.CSVWriterBuilder;
import com.opencsv.ICSVWriter;
import fr.gouv.vitam.common.LocalDateUtil;
import fr.gouv.vitam.common.client.VitamContext;
import fr.gouv.vitam.common.exception.VitamClientException;
import fr.gouv.vitamui.archives.search.common.common.RulesUpdateCommonService;
import fr.gouv.vitamui.archives.search.common.dto.ArchiveUnit;
import fr.gouv.vitamui.archives.search.common.dto.ArchiveUnitCsv;
import fr.gouv.vitamui.commons.api.domain.AgencyModelDto;
import fr.gouv.vitamui.commons.api.dtos.ExportSearchResultParam;
import fr.gouv.vitamui.commons.api.dtos.SearchCriteriaDto;
import fr.gouv.vitamui.commons.api.exception.BadRequestException;
import fr.gouv.vitamui.commons.api.exception.InvalidTypeException;
import fr.gouv.vitamui.commons.api.exception.RequestEntityTooLargeException;
import fr.gouv.vitamui.commons.vitam.api.dto.ResultsDto;
import fr.gouv.vitamui.commons.vitam.api.dto.VitamUISearchResponseDto;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Service;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static fr.gouv.vitamui.commons.api.utils.MetadataSearchCriteriaUtils.cleanString;
import static fr.gouv.vitamui.commons.api.utils.MetadataSearchCriteriaUtils.mapRequestToDslQuery;

/**
 * Archive-Search export service.
 */
@Service
public class ArchiveSearchUnitExportCsvService {

    private static final Logger LOGGER = LoggerFactory.getLogger(ArchiveSearchUnitExportCsvService.class);

    private static final Integer EXPORT_ARCHIVE_UNITS_MAX_ELEMENTS = 10000;
    public static final String FILING_UNIT = "FILING_UNIT";
    public static final String HOLDING_UNIT = "HOLDING_UNIT";
    private static final String INGEST_ARCHIVE_TYPE = "INGEST";

    private final ArchiveSearchAgenciesService archiveSearchAgenciesService;
    private final ArchiveSearchService archiveSearchService;

    private final ArchiveSearchThresholdService archiveSearchThresholdService;

    private final ArchiveSearchExternalParametersService archiveSearchExternalParametersService;
    private final ObjectMapper objectMapper;

    public ArchiveSearchUnitExportCsvService(
        final @Lazy ArchiveSearchService archiveSearchService,
        final ArchiveSearchAgenciesService archiveSearchAgenciesService,
        ArchiveSearchThresholdService archiveSearchThresholdService,
        ArchiveSearchExternalParametersService archiveSearchExternalParametersService,
        final ObjectMapper objectMapper
    ) {
        this.archiveSearchAgenciesService = archiveSearchAgenciesService;
        this.archiveSearchService = archiveSearchService;
        this.archiveSearchThresholdService = archiveSearchThresholdService;
        this.archiveSearchExternalParametersService = archiveSearchExternalParametersService;
        this.objectMapper = objectMapper;
    }

    /**
     * Export archive unit by criteria into csv file
     *
     * @param searchQuery
     * @throws VitamClientException
     * @throws IOException
     */
    public Resource exportToCsvSearchArchiveUnitsByCriteria(final SearchCriteriaDto searchQuery)
        throws VitamClientException {
        VitamContext vitamContext = archiveSearchExternalParametersService.buildVitamContextFromExternalParam();
        Optional<Long> thresholdOpt = archiveSearchThresholdService.retrieveProfilThresholds();
        thresholdOpt.ifPresent(searchQuery::setThreshold);

        LOGGER.info("Calling exportToCsvSearchArchiveUnitsByCriteria with query {} ", searchQuery);
        Locale locale = Locale.FRENCH;
        if (
            Locale.FRENCH.getLanguage().equals(searchQuery.getLanguage()) ||
            Locale.ENGLISH.getLanguage().equals(searchQuery.getLanguage())
        ) {
            locale = Locale.forLanguageTag(searchQuery.getLanguage());
        }
        ExportSearchResultParam exportSearchResultParam = new ExportSearchResultParam(locale);
        return exportToCsvSearchArchiveUnitsByCriteriaAndParams(searchQuery, exportSearchResultParam, vitamContext);
    }

    /**
     * export ToCsv Search ArchiveUnits By Criteria And Params by language
     *
     * @param searchQuery
     * @param exportSearchResultParam
     * @param vitamContext
     * @return
     * @throws VitamClientException
     */
    private Resource exportToCsvSearchArchiveUnitsByCriteriaAndParams(
        final SearchCriteriaDto searchQuery,
        final ExportSearchResultParam exportSearchResultParam,
        final VitamContext vitamContext
    ) throws VitamClientException {
        try {
            archiveSearchAgenciesService.mapAgenciesNameToCodes(searchQuery, vitamContext);
            List<ArchiveUnitCsv> unitCsvList = exportArchiveUnitsByCriteriaToCsvFile(searchQuery, vitamContext);
            // create a write
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
            Writer writer = new OutputStreamWriter(outputStream, StandardCharsets.UTF_8.name());
            // header record
            String[] headerRecordFr = exportSearchResultParam
                .getHeaders()
                .toArray(new String[exportSearchResultParam.getHeaders().size()]);
            SimpleDateFormat dateFormat = new SimpleDateFormat(exportSearchResultParam.getPatternDate());
            // create a csv writer
            ICSVWriter csvWriter = new CSVWriterBuilder(writer)
                .withSeparator(exportSearchResultParam.getSeparator())
                .withQuoteChar(ICSVWriter.DEFAULT_QUOTE_CHARACTER)
                .withEscapeChar(ICSVWriter.DEFAULT_ESCAPE_CHARACTER)
                .withLineEnd(ICSVWriter.DEFAULT_LINE_END)
                .build();
            // write header record
            csvWriter.writeNext(headerRecordFr);

            // write data records
            unitCsvList
                .stream()
                .forEach(archiveUnitCsv -> {
                    String startDt = null;
                    String endDt = null;
                    if (archiveUnitCsv.getStartDate() != null) {
                        try {
                            startDt = dateFormat.format(LocalDateUtil.getDate(archiveUnitCsv.getStartDate()));
                        } catch (ParseException e) {
                            LOGGER.error("Error parsing starting date {} ", archiveUnitCsv.getStartDate());
                        }
                    }
                    if (archiveUnitCsv.getEndDate() != null) {
                        try {
                            endDt = dateFormat.format(LocalDateUtil.getDate(archiveUnitCsv.getEndDate()));
                        } catch (ParseException e) {
                            LOGGER.error("Error parsing end date {} ", archiveUnitCsv.getEndDate());
                        }
                    }
                    csvWriter.writeNext(
                        new String[] {
                            archiveUnitCsv.getId(),
                            archiveUnitCsv.getArchiveUnitType(),
                            archiveUnitCsv.getOriginatingAgencyName(),
                            exportSearchResultParam.getDescriptionLevelMap().get(archiveUnitCsv.getDescriptionLevel()),
                            archiveUnitCsv.getTitle(),
                            startDt,
                            endDt,
                            archiveUnitCsv.getDescription(),
                        }
                    );
                });
            // close writers
            csvWriter.close();
            writer.close();
            return new ByteArrayResource(outputStream.toByteArray());
        } catch (IOException ex) {
            throw new BadRequestException("Unable to export csv file ", ex);
        }
    }

    private List<ArchiveUnitCsv> exportArchiveUnitsByCriteriaToCsvFile(
        final SearchCriteriaDto searchQuery,
        final VitamContext vitamContext
    ) throws VitamClientException {
        try {
            LOGGER.info("Calling exporting  export ArchiveUnits to CSV with criteria {}", searchQuery);
            checkSizeLimit(vitamContext, searchQuery);
            searchQuery.setPageNumber(0);
            searchQuery.setSize(EXPORT_ARCHIVE_UNITS_MAX_ELEMENTS);
            JsonNode archiveUnitsResult = archiveSearchService.searchArchiveUnits(
                mapRequestToDslQuery(searchQuery),
                vitamContext
            );
            final VitamUISearchResponseDto archivesResponse = objectMapper.treeToValue(
                archiveUnitsResult,
                VitamUISearchResponseDto.class
            );
            LOGGER.info("archivesResponse found {} ", archivesResponse.getResults().size());
            Set<String> originesAgenciesCodes = archivesResponse
                .getResults()
                .stream()
                .map(ResultsDto::getOriginatingAgency)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());

            List<AgencyModelDto> originAgenciesFound = archiveSearchAgenciesService.findOriginAgenciesByCodes(
                vitamContext,
                originesAgenciesCodes
            );
            Map<String, AgencyModelDto> agenciesMapByIdentifier = originAgenciesFound
                .stream()
                .collect(Collectors.toMap(AgencyModelDto::getIdentifier, agency -> agency));
            return archivesResponse
                .getResults()
                .stream()
                .map(
                    archiveUnit ->
                        RulesUpdateCommonService.fillOriginatingAgencyName(archiveUnit, agenciesMapByIdentifier)
                )
                .map(archiveUnit -> cleanAndMapArchiveUnitResult(archiveUnit, searchQuery.getLanguage()))
                .collect(Collectors.toList());
        } catch (IOException e) {
            LOGGER.error("Can't parse criteria as Vitam query {} : ", e);
            throw new BadRequestException("Can't parse criteria as Vitam query", e);
        }
    }

    /**
     * check limit of results limit
     *
     * @param vitamContext
     * @param searchQuery
     */
    private void checkSizeLimit(VitamContext vitamContext, SearchCriteriaDto searchQuery)
        throws VitamClientException, IOException {
        SearchCriteriaDto searchQueryCounting = new SearchCriteriaDto();
        searchQueryCounting.setCriteriaList(searchQuery.getCriteriaList());
        JsonNode archiveUnitsResult = archiveSearchService.searchArchiveUnits(
            mapRequestToDslQuery(searchQueryCounting),
            vitamContext
        );
        final VitamUISearchResponseDto archivesOriginResponse = objectMapper.treeToValue(
            archiveUnitsResult,
            VitamUISearchResponseDto.class
        );
        Integer nbResults = archivesOriginResponse.getHits().getTotal();
        if (nbResults >= EXPORT_ARCHIVE_UNITS_MAX_ELEMENTS) {
            LOGGER.error(
                "The archives units result found is greater than allowed {} ",
                EXPORT_ARCHIVE_UNITS_MAX_ELEMENTS
            );
            throw new RequestEntityTooLargeException(
                "The archives units result found is greater than allowed:  " + EXPORT_ARCHIVE_UNITS_MAX_ELEMENTS
            );
        }
    }

    private ArchiveUnitCsv cleanAndMapArchiveUnitResult(ArchiveUnit archiveUnit, String language) {
        if (archiveUnit == null) {
            return null;
        }
        ArchiveUnitCsv archiveUnitCsv = new ArchiveUnitCsv();
        BeanUtils.copyProperties(archiveUnit, archiveUnitCsv);
        archiveUnitCsv.setDescription(cleanString(getArchiveUnitDescription(archiveUnit)));
        archiveUnitCsv.setDescriptionLevel(
            archiveUnit.getDescriptionLevel() != null ? cleanString(archiveUnit.getDescriptionLevel()) : null
        );
        archiveUnitCsv.setArchiveUnitType(getArchiveUnitType(archiveUnit, language));
        archiveUnitCsv.setTitle(cleanString(getArchiveUnitTitle(archiveUnit)));
        archiveUnitCsv.setOriginatingAgencyName(
            archiveUnit.getOriginatingAgencyName() != null ? cleanString(archiveUnit.getOriginatingAgencyName()) : null
        );
        return archiveUnitCsv;
    }

    private String getArchiveUnitTitle(ArchiveUnit archiveUnit) {
        return getArchiveUnitI18nAttribute(archiveUnit, ResultsDto::getTitle, ResultsDto::getTitle_);
    }

    private String getArchiveUnitDescription(ArchiveUnit archiveUnit) {
        return getArchiveUnitI18nAttribute(archiveUnit, ResultsDto::getDescription, ResultsDto::getDescription_);
    }

    private String getArchiveUnitI18nAttribute(
        ArchiveUnit archiveUnit,
        Function<ArchiveUnit, String> attributeExtractor,
        Function<ArchiveUnit, Map<String, String>> i18nAttributeExtractor
    ) {
        if (archiveUnit == null) {
            return null;
        }
        final String attribute = attributeExtractor.apply(archiveUnit);
        if (StringUtils.isNotBlank(attribute)) {
            return attribute;
        }
        final Map<String, String> attribute_ = i18nAttributeExtractor.apply(archiveUnit);
        if (attribute_ == null) {
            return null;
        }
        return Stream.of("fr", "en")
            .map(lang -> attribute_.entrySet().stream().filter(e -> lang.equalsIgnoreCase(e.getKey())).findFirst())
            .flatMap(Optional::stream)
            .findFirst()
            .map(Map.Entry::getValue)
            .or(() -> attribute_.values().stream().findFirst())
            .orElse(null);
    }

    private String getArchiveUnitType(ArchiveUnit archiveUnit, String language) {
        String archiveUnitType = null;
        if (archiveUnit != null && !StringUtils.isEmpty(archiveUnit.getUnitType())) {
            switch (archiveUnit.getUnitType()) {
                case FILING_UNIT:
                    archiveUnitType = language.equals(Locale.FRENCH.getLanguage())
                        ? ExportSearchResultParam.FR_AU_FILING_SCHEME
                        : ExportSearchResultParam.EN_AU_FILING_SCHEME;
                    break;
                case HOLDING_UNIT:
                    archiveUnitType = language.equals(Locale.FRENCH.getLanguage())
                        ? ExportSearchResultParam.FR_AU_HOLDING_SCHEME
                        : ExportSearchResultParam.EN_AU_HOLDING_SCHEME;
                    break;
                case INGEST_ARCHIVE_TYPE:
                    if (StringUtils.isEmpty(archiveUnit.getUnitObject())) {
                        archiveUnitType = language.equals(Locale.FRENCH.getLanguage())
                            ? ExportSearchResultParam.FR_AU_WITHOUT_OBJECT
                            : ExportSearchResultParam.EN_AU_WITHOUT_OBJECT;
                    } else {
                        archiveUnitType = language.equals(Locale.FRENCH.getLanguage())
                            ? ExportSearchResultParam.FR_AU_WITH_OBJECT
                            : ExportSearchResultParam.EN_AU_WITH_OBJECT;
                    }
                    break;
                default:
                    throw new InvalidTypeException("Description Level Type is Unknown !");
            }
        }
        return archiveUnitType;
    }
}
