/*
 * Copyright French Prime minister Office/SGMAP/DINSIC/Vitam Program (2015-2022)
 *
 * contact.vitam@culture.gouv.fr
 *
 * This software is a computer program whose purpose is to implement a digital archiving back-office system managing
 * high volumetry securely and efficiently.
 *
 * This software is governed by the CeCILL 2.1 license under French law and abiding by the rules of distribution of free
 * software. You can use, modify and/ or redistribute the software under the terms of the CeCILL 2.1 license as
 * circulated by CEA, CNRS and INRIA at the following URL "https://cecill.info".
 *
 * As a counterpart to the access to the source code and rights to copy, modify and redistribute granted by the license,
 * users are provided only with a limited warranty and the software's author, the holder of the economic rights, and the
 * successive licensors have only limited liability.
 *
 * In this respect, the user's attention is drawn to the risks associated with loading, using, modifying and/or
 * developing or reproducing the software by the user in light of its specific status of free software, that may mean
 * that it is complicated to manipulate, and that also therefore means that it is reserved for developers and
 * experienced professionals having in-depth computer knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions enabling the security of their systems and/or data
 * to be ensured and, more generally, to use and operate it in the same conditions as regards security.
 *
 * The fact that you are presently reading this means that you have had knowledge of the CeCILL 2.1 license and that you
 * accept its terms.
 */

package fr.gouv.vitamui.collect.server.service;

import fr.gouv.vitam.collect.common.dto.BulkAtomicUpdateResult;
import fr.gouv.vitam.collect.external.client.CollectExternalClient;
import fr.gouv.vitam.common.client.VitamContext;
import fr.gouv.vitam.common.database.builder.request.multiple.UpdateMultiQuery;
import fr.gouv.vitam.common.exception.InvalidParseOperationException;
import fr.gouv.vitam.common.exception.VitamClientException;
import fr.gouv.vitam.common.json.JsonHandler;
import fr.gouv.vitam.common.model.RequestResponseOK;
import fr.gouv.vitamui.collect.common.dto.OperationIdDto;
import fr.gouv.vitamui.collect.common.dto.UpdateArchiveUnitDto;
import fr.gouv.vitamui.collect.common.dto.converter.UpdateArchiveUnitDtoToUpdateMultiQueryConverter;
import fr.gouv.vitamui.collect.common.exception.ArchiveUnitUpdateException;
import fr.gouv.vitamui.collect.common.model.OperationId;
import fr.gouv.vitamui.collect.common.service.ArchiveUnitService;
import fr.gouv.vitamui.commons.api.converter.JsonPatchDtoToUpdateMultiQueryConverter;
import fr.gouv.vitamui.commons.api.converter.UpdateMultiQueriesToBulkCommandDto;
import fr.gouv.vitamui.commons.api.dtos.BulkCommandDto;
import fr.gouv.vitamui.commons.api.dtos.JsonPatchDto;
import fr.gouv.vitamui.commons.api.dtos.MultiJsonPatchDto;
import lombok.RequiredArgsConstructor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import static fr.gouv.vitamui.commons.api.CommonConstants.X_REQUEST_ID_HEADER;

@Service
@RequiredArgsConstructor
public class ArchiveUnitServiceImpl implements ArchiveUnitService {

    private static final Logger log = LoggerFactory.getLogger(ArchiveUnitServiceImpl.class);
    private final CollectExternalClient collectExternalClient;
    private final UpdateArchiveUnitDtoToUpdateMultiQueryConverter updateArchiveUnitDtoToUpdateMultiQueryConverter;
    private final ExternalParametersService externalParametersService;
    private final JsonPatchDtoToUpdateMultiQueryConverter jsonPatchDtoToUpdateMultiQueryConverter;
    private final UpdateMultiQueriesToBulkCommandDto updateMultiQueriesToBulkCommandDto;

    @Override
    public OperationIdDto update(String transactionId, Set<UpdateArchiveUnitDto> updateArchiveUnitDtoSet) {
        final Set<UpdateMultiQuery> updateMultiQueries = updateArchiveUnitDtoSet
            .stream()
            .map(updateArchiveUnitDtoToUpdateMultiQueryConverter::convert)
            .filter(Objects::nonNull)
            .collect(Collectors.toSet());
        if (updateArchiveUnitDtoSet.size() != updateMultiQueries.size()) {
            throw new ArchiveUnitUpdateException("Fail to convert some archive unit updates payload to dsl queries");
        }
        return send(transactionId, updateMultiQueries);
    }

    @Override
    public OperationIdDto update(String transactionId, JsonPatchDto jsonPatchDto) {
        final UpdateMultiQuery updateMultiQuery = jsonPatchDtoToUpdateMultiQueryConverter.convert(jsonPatchDto);
        if (updateMultiQuery == null) {
            log.error("Failed to convert JSON patch to DSL query for archive unit: {}", jsonPatchDto.getId());
            throw new ArchiveUnitUpdateException(
                "Failed to convert JSON patch payload to DSL query. The patch may contain invalid field paths or operations."
            );
        }

        final Set<UpdateMultiQuery> updateMultiQueries = Set.of(updateMultiQuery);
        return send(transactionId, updateMultiQueries);
    }

    @Override
    public OperationIdDto update(String transactionId, MultiJsonPatchDto multiJsonPatchDto) {
        final Set<UpdateMultiQuery> updateMultiQueries = multiJsonPatchDto
            .stream()
            .map(jsonPatchDtoToUpdateMultiQueryConverter::convert)
            .filter(Objects::nonNull)
            .collect(Collectors.toSet());
        if (multiJsonPatchDto.size() != updateMultiQueries.size()) {
            throw new ArchiveUnitUpdateException("Fail to convert some json patch payloads to dsl queries");
        }
        return send(transactionId, updateMultiQueries);
    }

    private OperationIdDto send(String transactionId, Set<UpdateMultiQuery> updateMultiQueries) {
        final VitamContext context = externalParametersService.buildVitamContextFromExternalParam();
        final BulkCommandDto bulkCommandDto = updateMultiQueriesToBulkCommandDto.convert(updateMultiQueries);
        try {
            final RequestResponseOK<BulkAtomicUpdateResult> response = collectExternalClient.bulkAtomicUpdateUnits(
                context,
                transactionId,
                JsonHandler.toJsonNode(bulkCommandDto)
            );
            final OperationId operationId = new OperationId(response.getHeaderString(X_REQUEST_ID_HEADER));
            final OperationIdDto operationIdDto = new OperationIdDto().setOperationId(operationId);
            log.info("Operation started: {}", operationIdDto);
            return operationIdDto;
        } catch (VitamClientException | InvalidParseOperationException e) {
            log.error("Failed to update archive units for transaction: {}", transactionId, e);
            throw new ArchiveUnitUpdateException("Failed to update archive units", e);
        }
    }
}
