/*
 * Copyright French Prime minister Office/SGMAP/DINSIC/Vitam Program (2015-2022)
 *
 * contact.vitam@culture.gouv.fr
 *
 * This software is a computer program whose purpose is to implement a digital archiving back-office system managing
 * high volumetry securely and efficiently.
 *
 * This software is governed by the CeCILL 2.1 license under French law and abiding by the rules of distribution of free
 * software. You can use, modify and/ or redistribute the software under the terms of the CeCILL 2.1 license as
 * circulated by CEA, CNRS and INRIA at the following URL "https://cecill.info".
 *
 * As a counterpart to the access to the source code and rights to copy, modify and redistribute granted by the license,
 * users are provided only with a limited warranty and the software's author, the holder of the economic rights, and the
 * successive licensors have only limited liability.
 *
 * In this respect, the user's attention is drawn to the risks associated with loading, using, modifying and/or
 * developing or reproducing the software by the user in light of its specific status of free software, that may mean
 * that it is complicated to manipulate, and that also therefore means that it is reserved for developers and
 * experienced professionals having in-depth computer knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions enabling the security of their systems and/or data
 * to be ensured and, more generally, to use and operate it in the same conditions as regards security.
 *
 * The fact that you are presently reading this means that you have had knowledge of the CeCILL 2.1 license and that you
 * accept its terms.
 */
package fr.gouv.vitamui.collect.server.rest;

import fr.gouv.vitam.common.exception.VitamClientException;
import fr.gouv.vitamui.collect.common.dto.CollectProjectAttachmentsDto;
import fr.gouv.vitamui.collect.common.dto.CollectProjectConfigurationDto;
import fr.gouv.vitamui.collect.common.dto.CollectProjectContextDto;
import fr.gouv.vitamui.collect.common.dto.CollectProjectDescriptionDto;
import fr.gouv.vitamui.collect.common.dto.CollectProjectDto;
import fr.gouv.vitamui.collect.common.dto.CollectTransactionDto;
import fr.gouv.vitamui.collect.common.rest.RestApi;
import fr.gouv.vitamui.collect.server.service.ExternalParametersService;
import fr.gouv.vitamui.collect.server.service.ProjectService;
import fr.gouv.vitamui.common.security.SafeFileChecker;
import fr.gouv.vitamui.common.security.SanityChecker;
import fr.gouv.vitamui.commons.api.CommonConstants;
import fr.gouv.vitamui.commons.api.ParameterChecker;
import fr.gouv.vitamui.commons.api.domain.DirectionDto;
import fr.gouv.vitamui.commons.api.domain.PaginatedValuesDto;
import fr.gouv.vitamui.commons.api.domain.ServicesData;
import fr.gouv.vitamui.commons.api.exception.PreconditionFailedException;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.MediaType;
import org.springframework.security.access.annotation.Secured;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;

import java.io.InputStream;
import java.util.Optional;

import static fr.gouv.vitamui.collect.common.rest.RestApi.TRANSACTIONS;
import static fr.gouv.vitamui.commons.api.CommonConstants.LAST_TRANSACTION_PATH;
import static fr.gouv.vitamui.commons.api.CommonConstants.PATH_ID;

/**
 * Project External controller
 */
@RequestMapping(RestApi.COLLECT_PROJECT_PATH)
@RestController
@Tag(name = "Collect")
@ResponseBody
@RequiredArgsConstructor
public class ProjectController {

    private static final Logger LOGGER = LoggerFactory.getLogger(ProjectController.class);

    private static final String MANDATORY_IDENTIFIER = "The Identifier is a mandatory parameter: ";

    private final ProjectService projectService;
    private final ExternalParametersService externalParametersService;

    @Secured(ServicesData.ROLE_GET_PROJECTS)
    @GetMapping(value = "/paginated", params = { "page", "size" })
    public PaginatedValuesDto<CollectProjectDto> getAllPaginated(
        @RequestParam final Integer page,
        @RequestParam final Integer size,
        @RequestParam(required = false) final Optional<String> criteria,
        @RequestParam(required = false) final Optional<String> orderBy,
        @RequestParam(required = false) final Optional<DirectionDto> direction
    ) throws PreconditionFailedException {
        direction.ifPresent(SanityChecker::sanitizeCriteria);
        orderBy.ifPresent(SanityChecker::checkSecureParameter);
        SanityChecker.sanitizeCriteria(criteria);
        LOGGER.debug(
            "getPaginateEntities page={}, size={}, criteria={}, orderBy={}, ascendant={}",
            page,
            size,
            criteria,
            orderBy,
            direction
        );
        return projectService.getAllProjectsPaginated(
            page,
            size,
            criteria,
            orderBy,
            direction,
            externalParametersService.buildVitamContextFromExternalParam()
        );
    }

    @Operation(summary = "Get transactions by project paginated")
    @GetMapping(value = "/{id}" + TRANSACTIONS + "/paginated", params = { "page", "size" })
    @Secured(ServicesData.ROLE_GET_TRANSACTIONS)
    public PaginatedValuesDto<CollectTransactionDto> getTransactionsByProjectPaginated(
        @RequestParam final Integer page,
        @RequestParam final Integer size,
        @RequestParam(required = false) final Optional<String> criteria,
        @RequestParam(required = false) final Optional<String> orderBy,
        @RequestParam(required = false) final Optional<DirectionDto> direction,
        @PathVariable("id") String projectId
    ) throws PreconditionFailedException, VitamClientException {
        ParameterChecker.checkParameter(MANDATORY_IDENTIFIER, projectId);
        SanityChecker.checkSecureParameter(projectId);
        SanityChecker.sanitizeCriteria(direction);
        SanityChecker.sanitizeCriteria(criteria);
        orderBy.ifPresent(SanityChecker::checkSecureParameter);
        LOGGER.debug(
            "getTransactionsByProjectPaginated page={}, size={}, criteria={}, orderBy={}, direction={}",
            page,
            size,
            criteria,
            orderBy,
            direction
        );
        return projectService.getTransactionsByProjectPaginated(
            projectId,
            page,
            size,
            orderBy,
            direction,
            externalParametersService.buildVitamContextFromExternalParam()
        );
    }

    @Secured(ServicesData.ROLE_CREATE_PROJECTS)
    @PostMapping
    public CollectProjectDto createProject(@RequestBody CollectProjectDto collectProjectDto)
        throws PreconditionFailedException {
        ParameterChecker.checkParameter("the project is mandatory : ", collectProjectDto);
        SanityChecker.sanitizeCriteria(collectProjectDto);
        LOGGER.debug("Project to create : {}", collectProjectDto);
        return projectService.createProject(
            collectProjectDto,
            externalParametersService.buildVitamContextFromExternalParam()
        );
    }

    @Secured(ServicesData.ROLE_CREATE_TRANSACTIONS)
    @PostMapping(value = PATH_ID + "/transactions")
    public CollectTransactionDto createTransactionForProject(
        final @PathVariable("id") String id,
        @RequestBody CollectTransactionDto collectTransactionDto
    ) throws PreconditionFailedException {
        ParameterChecker.checkParameter(MANDATORY_IDENTIFIER, id);
        ParameterChecker.checkParameter("The transaction is a mandatory parameter: ", collectTransactionDto);
        SanityChecker.checkSecureParameter(id);
        SanityChecker.sanitizeCriteria(collectTransactionDto);
        LOGGER.debug("Transaction to create : {}", collectTransactionDto);
        return projectService.createTransactionForProject(
            collectTransactionDto,
            id,
            externalParametersService.buildVitamContextFromExternalParam()
        );
    }

    @Secured(ServicesData.ROLE_CREATE_PROJECTS)
    @Operation(summary = "Upload and stream collect zip file")
    @PostMapping(value = "/upload", consumes = MediaType.APPLICATION_OCTET_STREAM_VALUE)
    public void streamingUpload(
        InputStream inputStream,
        @RequestHeader(value = CommonConstants.X_TRANSACTION_ID_HEADER) final String transactionId,
        @RequestHeader(value = CommonConstants.X_ATTACHMENT_ID_HEADER, required = false) final String attachmentId,
        @RequestHeader(value = CommonConstants.X_ORIGINAL_FILENAME_HEADER) final String originalFileName
    ) throws PreconditionFailedException {
        ParameterChecker.checkParameter("The transaction ID is a mandatory parameter: ", transactionId);
        SanityChecker.checkSecureParameter(transactionId, attachmentId);
        SanityChecker.isValidFileName(originalFileName);
        SafeFileChecker.checkSafeFilePath(originalFileName);
        LOGGER.debug("[External] upload collect zip file : {}", originalFileName);
        projectService.streamingUpload(
            inputStream,
            transactionId,
            attachmentId,
            originalFileName,
            externalParametersService.buildVitamContextFromExternalParam()
        );
    }

    @Secured(ServicesData.ROLE_CREATE_PROJECTS)
    @Operation(summary = "Upload and stream SIP")
    @PostMapping(value = "/uploadSip", consumes = MediaType.APPLICATION_OCTET_STREAM_VALUE)
    public void streamingUploadSip(
        InputStream inputStream,
        @RequestHeader(value = CommonConstants.X_TRANSACTION_ID_HEADER) final String transactionId
    ) throws PreconditionFailedException {
        ParameterChecker.checkParameter("The transaction ID is a mandatory parameter: ", transactionId);
        SanityChecker.checkSecureParameter(transactionId);
        LOGGER.debug("[External] upload SIP");
        projectService.streamingUploadSip(
            inputStream,
            transactionId,
            externalParametersService.buildVitamContextFromExternalParam()
        );
    }

    @Secured(ServicesData.ROLE_UPDATE_PROJECTS_DESCRIPTION)
    @PutMapping(PATH_ID + "/description")
    public CollectProjectDto updateProjectDescription(
        final @PathVariable("id") String id,
        @RequestBody CollectProjectDescriptionDto collectProjectDescriptionDto
    ) throws PreconditionFailedException {
        ParameterChecker.checkParameter(MANDATORY_IDENTIFIER, id);
        SanityChecker.checkSecureParameter(id);
        SanityChecker.sanitizeCriteria(collectProjectDescriptionDto);
        LOGGER.debug("[External] Project description to update : {}", collectProjectDescriptionDto);
        return projectService.updateDescription(
            id,
            collectProjectDescriptionDto,
            externalParametersService.buildVitamContextFromExternalParam()
        );
    }

    @Secured(ServicesData.ROLE_UPDATE_PROJECTS_CONTEXT)
    @PutMapping(PATH_ID + "/context")
    public CollectProjectDto updateProjectContext(
        final @PathVariable("id") String id,
        @RequestBody CollectProjectContextDto collectProjectContextDto
    ) throws PreconditionFailedException {
        ParameterChecker.checkParameter(MANDATORY_IDENTIFIER, id);
        SanityChecker.checkSecureParameter(id);
        SanityChecker.sanitizeCriteria(collectProjectContextDto);
        LOGGER.debug("[External] Project context to update : {}", collectProjectContextDto);
        return projectService.updateContext(
            id,
            collectProjectContextDto,
            externalParametersService.buildVitamContextFromExternalParam()
        );
    }

    @Secured(ServicesData.ROLE_UPDATE_PROJECTS_ATTACHMENT)
    @PutMapping(PATH_ID + "/attachments")
    public CollectProjectDto updateProjectAttachments(
        final @PathVariable("id") String id,
        @RequestBody CollectProjectAttachmentsDto collectProjectAttachmentsDto
    ) throws PreconditionFailedException {
        ParameterChecker.checkParameter(MANDATORY_IDENTIFIER, id);
        SanityChecker.checkSecureParameter(id);
        SanityChecker.sanitizeCriteria(collectProjectAttachmentsDto);
        LOGGER.debug("[External] Project attachments to update : {}", collectProjectAttachmentsDto);
        return projectService.updateAttachments(
            id,
            collectProjectAttachmentsDto,
            externalParametersService.buildVitamContextFromExternalParam()
        );
    }

    @Secured(ServicesData.ROLE_UPDATE_PROJECTS_CONFIG)
    @PutMapping(PATH_ID + "/configuration")
    public CollectProjectDto updateProjectConfiguration(
        final @PathVariable("id") String id,
        @RequestBody CollectProjectConfigurationDto collectProjectConfigurationDto
    ) throws PreconditionFailedException {
        ParameterChecker.checkParameter(MANDATORY_IDENTIFIER, id);
        SanityChecker.checkSecureParameter(id);
        SanityChecker.sanitizeCriteria(collectProjectConfigurationDto);
        LOGGER.debug("[External] Project configuration to update : {}", collectProjectConfigurationDto);
        return projectService.updateConfiguration(
            id,
            collectProjectConfigurationDto,
            externalParametersService.buildVitamContextFromExternalParam()
        );
    }

    @Secured(ServicesData.ROLE_GET_PROJECTS)
    @GetMapping(PATH_ID)
    public CollectProjectDto findProjectById(final @PathVariable("id") String id)
        throws PreconditionFailedException, VitamClientException {
        ParameterChecker.checkParameter(MANDATORY_IDENTIFIER, id);
        SanityChecker.checkSecureParameter(id);
        LOGGER.debug("The project id {} ", id);
        return projectService.getProjectById(id, externalParametersService.buildVitamContextFromExternalParam());
    }

    @Secured(ServicesData.ROLE_DELETE_PROJECTS)
    @DeleteMapping(PATH_ID)
    public void deleteProjectById(final @PathVariable("id") String id)
        throws PreconditionFailedException, VitamClientException {
        ParameterChecker.checkParameter(MANDATORY_IDENTIFIER, id);
        SanityChecker.checkSecureParameter(id);
        LOGGER.debug("The project id {} ", id);
        projectService.deleteProjectById(id, externalParametersService.buildVitamContextFromExternalParam());
    }

    @Secured(ServicesData.ROLE_GET_TRANSACTIONS)
    @GetMapping(PATH_ID + LAST_TRANSACTION_PATH)
    public CollectTransactionDto findLastTransactionByProjectId(final @PathVariable("id") String id)
        throws PreconditionFailedException, VitamClientException {
        ParameterChecker.checkParameter(MANDATORY_IDENTIFIER, id);
        SanityChecker.checkSecureParameter(id);
        LOGGER.debug("Find the transaction by project with ID {}", id);
        return projectService.getLastTransactionForProjectId(
            id,
            externalParametersService.buildVitamContextFromExternalParam()
        );
    }
}
