/**
 * Copyright French Prime minister Office/SGMAP/DINSIC/Vitam Program (2019-2020)
 * and the signatories of the "VITAM - Accord du Contributeur" agreement.
 *
 * contact@programmevitam.fr
 *
 * This software is a computer program whose purpose is to implement
 * implement a digital archiving front-office system for the secure and
 * efficient high volumetry VITAM solution.
 *
 * This software is governed by the CeCILL-C license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL-C
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided only
 * with a limited warranty  and the software's author,  the holder of the
 * economic rights,  and the successive licensors  have only  limited
 * liability.
 *
 * In this respect, the user's attention is drawn to the risks associated
 * with loading,  using,  modifying and/or developing or reproducing the
 * software by the user in light of its specific status of free software,
 * that may mean  that it is complicated to manipulate,  and  that  also
 * therefore means  that it is reserved for developers  and  experienced
 * professionals having in-depth computer knowledge. Users are therefore
 * encouraged to load and test the software's suitability as regards their
 * requirements in conditions enabling the security of their systems and/or
 * data to be ensured and,  more generally, to use and operate it in the
 * same conditions as regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL-C license and that you accept its terms.
 */
package fr.gouv.vitamui.iam.server.user.service;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import fr.gouv.vitam.common.client.VitamContext;
import fr.gouv.vitam.common.exception.VitamClientException;
import fr.gouv.vitam.common.model.RequestResponse;
import fr.gouv.vitam.common.model.logbook.LogbookOperation;
import fr.gouv.vitamui.commons.api.CommonConstants;
import fr.gouv.vitamui.commons.api.converter.Converter;
import fr.gouv.vitamui.commons.api.domain.ApplicationDto;
import fr.gouv.vitamui.commons.api.domain.Criterion;
import fr.gouv.vitamui.commons.api.domain.CriterionOperator;
import fr.gouv.vitamui.commons.api.domain.GroupDto;
import fr.gouv.vitamui.commons.api.domain.IdDto;
import fr.gouv.vitamui.commons.api.domain.ProfileDto;
import fr.gouv.vitamui.commons.api.domain.QueryDto;
import fr.gouv.vitamui.commons.api.domain.QueryOperator;
import fr.gouv.vitamui.commons.api.domain.ServicesData;
import fr.gouv.vitamui.commons.api.domain.TenantDto;
import fr.gouv.vitamui.commons.api.domain.TenantInformationDto;
import fr.gouv.vitamui.commons.api.domain.UserDto;
import fr.gouv.vitamui.commons.api.domain.UserInfoDto;
import fr.gouv.vitamui.commons.api.enums.UserStatusEnum;
import fr.gouv.vitamui.commons.api.enums.UserTypeEnum;
import fr.gouv.vitamui.commons.api.exception.ApplicationServerException;
import fr.gouv.vitamui.commons.api.exception.ForbiddenException;
import fr.gouv.vitamui.commons.api.exception.InternalServerException;
import fr.gouv.vitamui.commons.api.exception.NotFoundException;
import fr.gouv.vitamui.commons.api.exception.NotImplementedException;
import fr.gouv.vitamui.commons.api.utils.CastUtils;
import fr.gouv.vitamui.commons.api.utils.EnumUtils;
import fr.gouv.vitamui.commons.logbook.common.EventType;
import fr.gouv.vitamui.commons.logbook.dto.EventDiffDto;
import fr.gouv.vitamui.commons.mongo.service.SequenceGeneratorService;
import fr.gouv.vitamui.commons.security.client.config.password.PasswordConfiguration;
import fr.gouv.vitamui.commons.security.client.dto.AuthUserDto;
import fr.gouv.vitamui.commons.security.client.dto.BasicCustomerDto;
import fr.gouv.vitamui.commons.security.client.dto.GraphicIdentityDto;
import fr.gouv.vitamui.commons.utils.JsonUtils;
import fr.gouv.vitamui.commons.utils.VitamUIUtils;
import fr.gouv.vitamui.commons.vitam.api.access.LogbookService;
import fr.gouv.vitamui.commons.vitam.api.dto.LogbookEventDto;
import fr.gouv.vitamui.commons.vitam.api.dto.LogbookOperationDto;
import fr.gouv.vitamui.commons.vitam.api.dto.LogbookOperationsCommonResponseDto;
import fr.gouv.vitamui.commons.vitam.api.util.VitamRestUtils;
import fr.gouv.vitamui.iam.common.enums.OtpEnum;
import fr.gouv.vitamui.iam.common.utils.IamUtils;
import fr.gouv.vitamui.iam.security.service.SecurityService;
import fr.gouv.vitamui.iam.server.application.service.ApplicationService;
import fr.gouv.vitamui.iam.server.common.ApiIamConstants;
import fr.gouv.vitamui.iam.server.common.domain.Address;
import fr.gouv.vitamui.iam.server.common.domain.MongoDbCollections;
import fr.gouv.vitamui.iam.server.common.domain.SequencesConstants;
import fr.gouv.vitamui.iam.server.common.service.AddressService;
import fr.gouv.vitamui.iam.server.customer.dao.CustomerRepository;
import fr.gouv.vitamui.iam.server.customer.domain.Customer;
import fr.gouv.vitamui.iam.server.group.service.GroupService;
import fr.gouv.vitamui.iam.server.logbook.service.IamLogbookService;
import fr.gouv.vitamui.iam.server.profile.service.ProfileService;
import fr.gouv.vitamui.iam.server.security.AbstractResourceClientService;
import fr.gouv.vitamui.iam.server.tenant.dao.TenantRepository;
import fr.gouv.vitamui.iam.server.tenant.domain.Tenant;
import fr.gouv.vitamui.iam.server.user.converter.UserConverter;
import fr.gouv.vitamui.iam.server.user.dao.UserRepository;
import fr.gouv.vitamui.iam.server.user.domain.AlertAnalytics;
import fr.gouv.vitamui.iam.server.user.domain.User;
import lombok.Getter;
import lombok.Setter;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.bson.Document;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.Resource;
import org.springframework.data.mongodb.MongoTransactionManager;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.CriteriaDefinition;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.springframework.util.Assert;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.time.OffsetDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static fr.gouv.vitamui.commons.api.CommonConstants.APPLICATION_ID;
import static fr.gouv.vitamui.commons.api.CommonConstants.GPDR_DEFAULT_VALUE;
import static fr.gouv.vitamui.commons.api.CommonConstants.USER_ID_ATTRIBUTE;
import static fr.gouv.vitamui.commons.logbook.common.EventType.EXT_VITAMUI_BLOCK_USER;
import static fr.gouv.vitamui.commons.logbook.common.EventType.EXT_VITAMUI_CREATE_USER;
import static fr.gouv.vitamui.commons.logbook.common.EventType.EXT_VITAMUI_CREATE_USER_INFO;
import static fr.gouv.vitamui.commons.logbook.common.EventType.EXT_VITAMUI_PASSWORD_CHANGE;
import static fr.gouv.vitamui.commons.logbook.common.EventType.EXT_VITAMUI_PASSWORD_INIT;
import static fr.gouv.vitamui.commons.logbook.common.EventType.EXT_VITAMUI_PASSWORD_REVOCATION;
import static fr.gouv.vitamui.commons.logbook.common.EventType.EXT_VITAMUI_UPDATE_USER;
import static fr.gouv.vitamui.commons.logbook.common.EventType.EXT_VITAMUI_UPDATE_USER_INFO;

/**
 * The service to read, create, update and delete the users.
 */
@Getter
@Setter
public class UserService extends AbstractResourceClientService<UserDto, User> {

    private static final Logger LOGGER = LoggerFactory.getLogger(UserService.class);

    private static final String TYPE_KEY = "type";
    private static final String LEVEL_KEY = "level";
    public static final int MAX_OLD_PASSWORDS = 12;
    private final String ADMIN_EMAIL_PATTERN = "admin@";
    private final String PORTAL_APP_IDENTIFIER = "PORTAL_APP";

    private static final List<EventType> USER_OPERATIONS_EVENT_TYPES = List.of(
        EXT_VITAMUI_CREATE_USER,
        EXT_VITAMUI_UPDATE_USER,
        EXT_VITAMUI_BLOCK_USER,
        EXT_VITAMUI_PASSWORD_REVOCATION,
        EXT_VITAMUI_PASSWORD_INIT,
        EXT_VITAMUI_PASSWORD_CHANGE,
        EXT_VITAMUI_CREATE_USER_INFO,
        EXT_VITAMUI_UPDATE_USER_INFO
    );

    private final UserRepository userRepository;
    private final GroupService groupService;
    private final ProfileService profileService;
    private final UserEmailService userEmailService;
    private final TenantRepository tenantRepository;
    private final SecurityService securityService;
    private final CustomerRepository customerRepository;
    private final IamLogbookService iamLogbookService;
    private final UserConverter userConverter;
    private final MongoTransactionManager mongoTransactionManager;
    private final LogbookService logbookService;
    private final AddressService addressService;
    private final ApplicationService applicationService;
    private final Integer maxOldPassword;
    private final UserExportService userExportService;
    private final UserInfoService userInfoService;
    private final ConnectionHistoryService connectionHistoryService;

    @Autowired
    public UserService(
        final SequenceGeneratorService sequenceGeneratorService,
        final UserRepository userRepository,
        final GroupService groupService,
        final ProfileService profileService,
        final UserEmailService userEmailService,
        final TenantRepository tenantRepository,
        final SecurityService securityService,
        final CustomerRepository customerRepository,
        final IamLogbookService iamLogbookService,
        final UserConverter userConverter,
        final MongoTransactionManager mongoTransactionManager,
        final LogbookService logbookService,
        final AddressService addressService,
        final ApplicationService applicationService,
        final PasswordConfiguration passwordConfiguration,
        final UserExportService userExportService,
        final UserInfoService userInfoService,
        final ConnectionHistoryService connectionHistoryService
    ) {
        super(sequenceGeneratorService, securityService);
        this.userRepository = userRepository;
        this.groupService = groupService;
        this.profileService = profileService;
        this.userEmailService = userEmailService;
        this.tenantRepository = tenantRepository;
        this.securityService = securityService;
        this.customerRepository = customerRepository;
        this.iamLogbookService = iamLogbookService;
        this.userConverter = userConverter;
        this.mongoTransactionManager = mongoTransactionManager;
        this.logbookService = logbookService;
        this.addressService = addressService;
        this.applicationService = applicationService;
        this.maxOldPassword = (passwordConfiguration != null && passwordConfiguration.getMaxOldPassword() != null)
            ? passwordConfiguration.getMaxOldPassword()
            : MAX_OLD_PASSWORDS;
        this.userExportService = userExportService;
        this.userInfoService = userInfoService;
        this.connectionHistoryService = connectionHistoryService;
    }

    @Override
    protected void checkAllowedOrderby(Optional<String> optOrderBy) {
        optOrderBy.ifPresent(orderBy -> {
            if (orderBy.trim().equalsIgnoreCase("password")) {
                throw new ForbiddenException("forbidden orderby field");
            }
        });
    }

    /**
     * This method must be only used by the Authentification Service during the authentication process
     */
    public UserDto findUserById(final String id) {
        return super.getOneByPassSecurity(id, Optional.empty());
    }

    public UserDto findUserByEmailAndCustomerId(final String email, final String customerId) {
        final User user = getRepository().findByEmailIgnoreCaseAndCustomerId(email, customerId);
        if (user == null) {
            throw new NotFoundException("User not found for email: " + email);
        }
        return convertFromEntityToDto(user);
    }

    public List<UserDto> findUsersByEmail(final String email) {
        List<User> users = getRepository().findAllByEmailIgnoreCase(email);
        return users.stream().map(this::convertFromEntityToDto).collect(Collectors.toList());
    }

    public AuthUserDto getMe() {
        return securityService.getUser();
    }

    @Override
    protected void beforeCreate(final UserDto dto) {
        final String message = "Unable to create user " + dto.getEmail() + " (" + dto.getCustomerId() + ")";

        checkSetReadonly(dto.isReadonly(), message);
        checkCustomer(dto.getCustomerId(), message);
        checkEmail(dto.getEmail(), dto.getCustomerId(), message);
        checkGroupId(dto.getGroupId(), message);
        super.checkIdentifier(dto.getIdentifier(), message);

        final GroupDto groupDto = getGroupDtoById(dto.getGroupId(), message);
        checkGroup(groupDto, dto.getCustomerId(), message);
        checkLevel(groupDto.getLevel(), message);
        checkOtp(dto);

        if (dto.getPhone() != null) {
            checkPhoneNumber(dto.getPhone());
        }

        if (dto.getMobile() != null) {
            checkPhoneNumber(dto.getMobile());
        }

        dto.setLevel(groupDto.getLevel());
        dto.setIdentifier(getNextSequenceId(SequencesConstants.USER_IDENTIFIER));
        dto.setPasswordExpirationDate(getPasswordExpirationDate(dto.getCustomerId()));
        dto.setEmail(dto.getEmail().toLowerCase());
    }

    public Resource exportUsers(final Optional<String> criteria) {
        try (final var xlsOutputStream = new ByteArrayOutputStream()) {
            final List<UserDto> usersDto = this.getAll(criteria);
            final List<String> userIds = usersDto.stream().map(UserDto::getIdentifier).collect(Collectors.toList());
            final List<String> userInfoIds = usersDto.stream().map(UserDto::getUserInfoId).collect(Collectors.toList());
            final List<String> userGroupIds = usersDto.stream().map(UserDto::getGroupId).collect(Collectors.toList());

            final LogbookOperationsCommonResponseDto userOperations = getUserOperations(getIdentifiers(usersDto));
            final LogbookOperationsCommonResponseDto userInfoOperations = getUserInfoOperations(
                getIdentifiers(usersDto)
            );

            final List<LogbookEventDto> userEvents = mapToEvents(userOperations, userInfoOperations);
            final List<LogbookEventDto> filteredUserEvents = filterUserEvents(userEvents, userIds);

            userExportService.createXlsxFile(
                usersDto,
                filteredUserEvents,
                buildUsersInfoLangMap(userInfoIds),
                buildUsersGroupNamesMap(userGroupIds),
                xlsOutputStream
            );

            return new ByteArrayResource(xlsOutputStream.toByteArray());
        } catch (final IOException exception) {
            throw new InternalServerException("An error occurred while creating the xls users list export", exception);
        }
    }

    private List<LogbookEventDto> filterUserEvents(final List<LogbookEventDto> userEvents, final List<String> userIds) {
        return userEvents
            .stream()
            .filter(
                logbookEventDto -> USER_OPERATIONS_EVENT_TYPES.contains(EventType.valueOf(logbookEventDto.getEvType()))
            )
            .filter(logbookEventDto -> userIds.contains(logbookEventDto.getObId()))
            .collect(Collectors.toList());
    }

    private List<String> getIdentifiers(List<UserDto> usersDto) {
        return usersDto.stream().map(UserDto::getIdentifier).toList();
    }

    private List<LogbookEventDto> mapToEvents(
        LogbookOperationsCommonResponseDto userOperations,
        LogbookOperationsCommonResponseDto userInfoOperations
    ) {
        final Stream<LogbookOperationDto> mergedOperations = Stream.concat(
            userOperations.getResults().stream(),
            userInfoOperations.getResults().stream()
        );

        return mergedOperations
            .map(operation -> {
                operation
                    .getEvents()
                    .forEach(logbookEventDto -> logbookEventDto.setEvIdAppSession(operation.getEvIdAppSession()));
                return operation.getEvents();
            })
            .flatMap(Collection::stream)
            .collect(Collectors.toList());
    }

    private LogbookOperationsCommonResponseDto getUserOperations(List<String> userIdentifiers) {
        VitamContext vitamContext = securityService.buildVitamContext(securityService.getTenantIdentifier());
        ObjectNode usersQuery = logbookService.buildQuery(userIdentifiers, MongoDbCollections.USERS);

        try {
            RequestResponse<LogbookOperation> usersLogbookOperations = logbookService.selectOperations(
                usersQuery,
                vitamContext
            );
            return VitamRestUtils.responseMapping(
                usersLogbookOperations.toJsonNode(),
                LogbookOperationsCommonResponseDto.class
            );
        } catch (VitamClientException exception) {
            throw new InternalServerException("An error occurred while fetching user operations", exception);
        }
    }

    private LogbookOperationsCommonResponseDto getUserInfoOperations(List<String> userIdentifiers) {
        VitamContext vitamContext = securityService.buildVitamContext(securityService.getTenantIdentifier());
        ObjectNode userInfoQuery = logbookService.buildQuery(userIdentifiers, MongoDbCollections.USER_INFOS);

        try {
            RequestResponse<LogbookOperation> userInfoLogbookOperations = logbookService.selectOperations(
                userInfoQuery,
                vitamContext
            );
            return VitamRestUtils.responseMapping(
                userInfoLogbookOperations.toJsonNode(),
                LogbookOperationsCommonResponseDto.class
            );
        } catch (VitamClientException exception) {
            throw new InternalServerException("An error occurred while fetching user operations", exception);
        }
    }

    /**
     * User Creation.
     * Email sent to user is not mandatory for user creation.
     * Also we can't use {@link Transactional} because before sending an email, CAS check user existence and with the transaction the user isn't processed yet.
     * {@inheritDoc}
     */
    @Override
    public UserDto create(final UserDto userDto) {
        UserDto createdUserDto = null;

        TransactionStatus status = null;
        if (mongoTransactionManager != null) {
            final TransactionDefinition definition = new DefaultTransactionDefinition(
                TransactionDefinition.PROPAGATION_REQUIRED
            );
            status = mongoTransactionManager.getTransaction(definition);
        }

        try {
            createdUserDto = super.create(userDto);
            iamLogbookService.createUserEvent(createdUserDto);

            if (mongoTransactionManager != null) {
                mongoTransactionManager.commit(status);
            }
        } catch (final Exception e) {
            if (mongoTransactionManager != null) {
                mongoTransactionManager.rollback(status);
            }
            throw e;
        }
        userEmailService.sendCreationEmail(createdUserDto);
        return createdUserDto;
    }

    protected OffsetDateTime getPasswordExpirationDate(final String customerId) {
        final Optional<Customer> customer = customerRepository.findById(customerId);
        Assert.isTrue(customer.isPresent(), "Customer does not exist");

        return OffsetDateTime.now().plusMonths(customer.get().getPasswordRevocationDelay());
    }

    @Override
    protected void beforeUpdate(final UserDto dto) {
        final String message = "Unable to update user " + dto.getId();
        final User user = find(dto.getId(), dto.getCustomerId(), message);

        checkIsReadonly(user.isReadonly(), message);
        checkCustomer(user.getCustomerId(), message);
        checkLevel(user.getLevel(), message);

        checkSetReadonly(dto.isReadonly(), message);
        if (!StringUtils.equalsIgnoreCase(user.getEmail(), dto.getEmail())) {
            checkEmail(dto.getEmail(), user.getCustomerId(), message);
        }
        checkSetReadonly(dto.isReadonly(), message);

        final GroupDto groupDto = getGroupDtoById(dto.getGroupId(), message);
        checkGroup(groupDto, user.getCustomerId(), message);
        checkLevel(groupDto.getLevel(), message);
        checkOtp(dto);

        if (dto.getPhone() != null && !StringUtils.equalsIgnoreCase(user.getPhone(), dto.getPhone())) {
            checkPhoneNumber(dto.getPhone());
        }

        if (dto.getMobile() != null && !StringUtils.equalsIgnoreCase(user.getMobile(), dto.getMobile())) {
            checkPhoneNumber(dto.getMobile());
        }

        dto.setLevel(groupDto.getLevel());
        dto.setIdentifier(user.getIdentifier());
        dto.setPasswordExpirationDate(user.getPasswordExpirationDate());
        dto.setEmail(dto.getEmail().toLowerCase());
    }

    /**
     * User Update.
     * We can't use {@link Transactional} because before sending an email, CAS check user existence and with the transaction the user isn't processed yet.
     * {@inheritDoc}
     */
    @Override
    public UserDto update(final UserDto dto) {
        boolean sendMail = false;
        UserDto updatedUser = null;

        TransactionStatus status = null;
        if (mongoTransactionManager != null) {
            final TransactionDefinition definition = new DefaultTransactionDefinition(
                TransactionDefinition.PROPAGATION_REQUIRED
            );
            status = mongoTransactionManager.getTransaction(definition);
        }

        try {
            final VitamContext vitamContext = securityService.buildVitamContext(securityService.getTenantIdentifier());
            if (vitamContext != null) {
                LOGGER.debug("Update User EvIdAppSession : {} ", vitamContext.getApplicationSessionId());
            }

            LOGGER.debug("Update {} {}", getObjectName(), dto);
            beforeUpdate(dto);
            final User entity = convertFromDtoToEntity(dto);
            final String entityId = entity.getId();
            final Optional<User> optExistingUser = getRepository().findById(entityId);
            Assert.isTrue(
                optExistingUser.isPresent(),
                "Unable to update " + getObjectName() + ": no entity found with id: " + entityId
            );

            final User existingUser = optExistingUser.get();
            entity.setPassword(existingUser.getPassword());
            entity.setOldPasswords(existingUser.getOldPasswords());

            final UserStatusEnum existingStatus = existingUser.getStatus();
            final UserStatusEnum newStatus = dto.getStatus();
            if (
                statusEquals(newStatus, UserStatusEnum.ENABLED) && statusEquals(existingStatus, UserStatusEnum.DISABLED)
            ) {
                saveCurrentPasswordInOldPasswords(entity, entity.getPassword(), maxOldPassword);
                entity.setPassword(null);
                entity.setPasswordExpirationDate(OffsetDateTime.now());
                entity.setNbFailedAttempts(0);
                sendMail = true;

                final AuthUserDto authUserDto = securityService.getUser();
                iamLogbookService.revokePasswordEvent(dto, authUserDto.getSuperUserIdentifier());
            }

            final User savedEntity = getRepository().save(entity);
            updatedUser = convertFromEntityToDto(savedEntity);

            if (mongoTransactionManager != null) {
                mongoTransactionManager.commit(status);
            }
        } catch (final Exception e) {
            if (mongoTransactionManager != null) {
                mongoTransactionManager.rollback(status);
            }
            throw e;
        }

        if (sendMail) {
            userEmailService.sendCreationEmail(updatedUser);
        }

        return updatedUser;
    }

    public void saveCurrentPasswordInOldPasswords(
        final User user,
        final String newPassword,
        final Integer maxOldPassword
    ) {
        if (StringUtils.isNotBlank(newPassword)) {
            List<String> oldPasswords = user.getOldPasswords();
            if (oldPasswords == null) {
                oldPasswords = new ArrayList<>();
            }
            oldPasswords.add(0, newPassword);
            if (oldPasswords.size() > maxOldPassword) {
                oldPasswords = oldPasswords.subList(0, maxOldPassword);
            }
            user.setOldPasswords(oldPasswords);
        }
    }

    private boolean statusEquals(final UserStatusEnum status, final UserStatusEnum expectedStatus) {
        return status != null && status.equals(expectedStatus);
    }

    /**
     * User Patch.
     * We can't use {@link Transactional} because before sending an email, CAS check user existence and with the transaction the user isn't processed yet.
     * {@inheritDoc}
     */
    @Override
    public UserDto patch(final Map<String, Object> partialDto) {
        boolean sendMail = false;
        UserDto dto = null;

        TransactionStatus status = null;
        if (mongoTransactionManager != null) {
            final TransactionDefinition definition = new DefaultTransactionDefinition(
                TransactionDefinition.PROPAGATION_REQUIRED
            );
            status = mongoTransactionManager.getTransaction(definition);
        }

        try {
            LOGGER.debug("Patch {} with {}", getObjectName(), partialDto);

            // replacing the email with the lowercase version during update
            final String email = CastUtils.toString(partialDto.get("email"));
            if (email != null) {
                partialDto.put("email", email.toLowerCase());
            }
            final User entity = beforePatch(partialDto);
            final UserStatusEnum existingStatus = entity.getStatus();
            processPatch(entity, partialDto);
            Assert.isTrue(
                getRepository().existsById(entity.getId()),
                "Unable to patch " + getObjectName() + ": no entity found with id: " + entity.getId()
            );

            final UserStatusEnum newStatus = entity.getStatus();
            if (
                statusEquals(existingStatus, UserStatusEnum.DISABLED) && statusEquals(newStatus, UserStatusEnum.ENABLED)
            ) {
                entity.setPassword(null);
                entity.setPasswordExpirationDate(OffsetDateTime.now());
                entity.setNbFailedAttempts(0);
                sendMail = true;

                final AuthUserDto authUserDto = securityService.getUser();
                iamLogbookService.revokePasswordEvent(entity, authUserDto.getSuperUserIdentifier());
            }

            final User savedEntity = getRepository().save(entity);
            dto = convertFromEntityToDto(savedEntity);

            if (mongoTransactionManager != null) {
                mongoTransactionManager.commit(status);
            }
        } catch (final Exception e) {
            if (mongoTransactionManager != null) {
                mongoTransactionManager.rollback(status);
            }
            throw e;
        }

        if (sendMail) {
            userEmailService.sendCreationEmail(dto);
        }
        return dto;
    }

    @Override
    protected User beforePatch(final Map<String, Object> partialDto) {
        final String id = CastUtils.toString(partialDto.get("id"));
        final String message = "Unable to patch user " + id;
        final String customerId = CastUtils.toString(partialDto.get("customerId"));
        final User user = find(id, customerId, message);

        Assert.isTrue(!partialDto.containsKey("password"), message + user.getId() + " : cannot patch password");
        Assert.isTrue(!partialDto.containsKey("identifier"), message + user.getId() + " cannot patch identifier");
        Assert.isTrue(!partialDto.containsKey("readonly"), message + user.getId() + " cannot patch readonly");
        Assert.isTrue(!partialDto.containsKey("level"), message + user.getId() + " cannot patch level");
        Assert.isTrue(
            !UserStatusEnum.BLOCKED.toString().equals(partialDto.get("status")),
            "User can't be blocked by API, this action need a special workflow for be realised"
        );
        Assert.isTrue(!checkMapContainsOnlyFieldsUnmodifiable(partialDto, Arrays.asList("id", "customerId")), message);

        checkLevel(user.getLevel(), message);
        checkIsReadonly(user.isReadonly(), message);
        final Boolean userHasOtp = CastUtils.toBoolean(partialDto.get("otp"));
        if (userHasOtp != null) {
            checkOtp(user, userHasOtp);
        }

        final String email = CastUtils.toString(partialDto.get("email"));
        if (email != null) {
            checkEmail(email, user.getCustomerId(), message);
        }

        final String groupId = CastUtils.toString(partialDto.get("groupId"));
        if (!StringUtils.isEmpty(groupId)) {
            final GroupDto groupDto = getGroupDtoById(groupId, message);
            checkGroup(groupDto, customerId, message);
            if (!StringUtils.equals(user.getLevel(), groupDto.getLevel())) {
                checkLevel(groupDto.getLevel(), message);
                partialDto.put("level", groupDto.getLevel());
            }
        }

        final String phone = CastUtils.toString(partialDto.get("phone"));
        final String mobile = CastUtils.toString(partialDto.get("mobile"));
        if (phone != null && !StringUtils.equalsIgnoreCase(user.getPhone(), phone)) {
            checkPhoneNumber(phone);
        }

        if (mobile != null && !StringUtils.equalsIgnoreCase(user.getMobile(), mobile)) {
            checkPhoneNumber(mobile);
        }

        return user;
    }

    @Override
    protected void processPatch(final User user, final Map<String, Object> partialDto) {
        final Collection<EventDiffDto> logbooks = new ArrayList<>();
        for (final Entry<String, Object> entry : partialDto.entrySet()) {
            switch (entry.getKey()) {
                case "id":
                case "customerId":
                    break;
                case "email":
                    logbooks.add(new EventDiffDto(UserConverter.EMAIL_KEY, GPDR_DEFAULT_VALUE, GPDR_DEFAULT_VALUE));
                    user.setEmail(CastUtils.toString(entry.getValue()));
                    break;
                case "firstname":
                    logbooks.add(new EventDiffDto(UserConverter.FIRSTNAME_KEY, GPDR_DEFAULT_VALUE, GPDR_DEFAULT_VALUE));
                    user.setFirstname(CastUtils.toString(entry.getValue()));
                    break;
                case "lastname":
                    logbooks.add(new EventDiffDto(UserConverter.LASTNAME_KEY, GPDR_DEFAULT_VALUE, GPDR_DEFAULT_VALUE));
                    user.setLastname(CastUtils.toString(entry.getValue()));
                    break;
                case "language":
                    logbooks.add(new EventDiffDto(UserConverter.LANGUAGE_KEY, user.getLanguage(), entry.getValue()));
                    user.setLanguage(CastUtils.toString(entry.getValue()));
                    break;
                case "type":
                    final String typeAsString = CastUtils.toString(entry.getValue());
                    logbooks.add(new EventDiffDto(UserConverter.TYPE_KEY, user.getType(), typeAsString));
                    user.setType(EnumUtils.stringToEnum(UserTypeEnum.class, typeAsString));
                    break;
                case "level":
                    logbooks.add(new EventDiffDto(UserConverter.LEVEL_KEY, user.getLevel(), entry.getValue()));
                    user.setLevel(CastUtils.toString(entry.getValue()));
                    break;
                case "mobile":
                    logbooks.add(new EventDiffDto(UserConverter.MOBILE_KEY, GPDR_DEFAULT_VALUE, GPDR_DEFAULT_VALUE));
                    user.setMobile(CastUtils.toString(entry.getValue()));
                    break;
                case "phone":
                    logbooks.add(new EventDiffDto(UserConverter.PHONE_KEY, GPDR_DEFAULT_VALUE, GPDR_DEFAULT_VALUE));
                    user.setPhone(CastUtils.toString(entry.getValue()));
                    break;
                case "groupId":
                    final GroupDto oldGroup = groupService.getOne(
                        user.getGroupId(),
                        Optional.empty(),
                        Optional.empty()
                    );
                    if (CastUtils.toString(entry.getValue()).isEmpty()) {
                        logbooks.add(
                            new EventDiffDto(
                                UserConverter.GROUP_IDENTIFIER_KEY,
                                oldGroup.getIdentifier(),
                                Optional.empty()
                            )
                        );
                        user.setGroupId(CastUtils.toString(entry.getValue()));
                    } else {
                        final GroupDto newGroup = groupService.getOne(
                            CastUtils.toString(entry.getValue()),
                            Optional.empty(),
                            Optional.empty()
                        );
                        logbooks.add(
                            new EventDiffDto(
                                UserConverter.GROUP_IDENTIFIER_KEY,
                                oldGroup.getIdentifier(),
                                newGroup.getIdentifier()
                            )
                        );
                        user.setGroupId(CastUtils.toString(entry.getValue()));
                    }
                    break;
                case "status":
                    final String status = CastUtils.toString(entry.getValue());
                    logbooks.add(new EventDiffDto(UserConverter.STATUS_KEY, user.getStatus(), status));
                    user.setStatus(EnumUtils.stringToEnum(UserStatusEnum.class, status));

                    if (user.getStatus() == UserStatusEnum.DISABLED) {
                        logbooks.add(
                            new EventDiffDto(
                                UserConverter.DISABLING_DATE,
                                user.getDisablingDate(),
                                OffsetDateTime.now()
                            )
                        );
                        user.setDisablingDate(OffsetDateTime.now());
                    }
                    if (user.getStatus() == UserStatusEnum.REMOVED) {
                        logbooks.add(
                            new EventDiffDto(UserConverter.REMOVING_DATE, user.getRemovingDate(), OffsetDateTime.now())
                        );
                        user.setRemovingDate(OffsetDateTime.now());
                        user.setDisablingDate(null);
                        connectionHistoryService.deleteByUserId(user.getId());
                    }
                    if (user.getStatus() == UserStatusEnum.ENABLED) {
                        user.setDisablingDate(null);
                        user.setRemovingDate(null);
                        user.setNbFailedAttempts(0);
                    }

                    break;
                case "subrogeable":
                    logbooks.add(
                        new EventDiffDto(UserConverter.SUBROGEABLE_KEY, user.isSubrogeable(), entry.getValue())
                    );
                    user.setSubrogeable(CastUtils.toBoolean(entry.getValue()));
                    break;
                case "otp":
                    logbooks.add(new EventDiffDto(UserConverter.OTP_KEY, user.isOtp(), entry.getValue()));
                    user.setOtp(CastUtils.toBoolean(entry.getValue()));
                    break;
                case "address":
                    final Address address = user.getAddress();
                    if (address == null) {
                        user.setAddress(new Address());
                    }
                    addressService.processPatch(user.getAddress(), CastUtils.toMap(entry.getValue()), logbooks, true);
                    break;
                case "internalCode":
                    logbooks.add(
                        new EventDiffDto(UserConverter.INTERNAL_CODE_KEY, user.getInternalCode(), entry.getValue())
                    );
                    user.setInternalCode(CastUtils.toString(entry.getValue()));
                    break;
                case "siteCode":
                    logbooks.add(new EventDiffDto(UserConverter.SITE_CODE, user.getSiteCode(), entry.getValue()));
                    user.setSiteCode(CastUtils.toString(entry.getValue()));
                    break;
                case "centerCodes":
                    logbooks.add(new EventDiffDto(UserConverter.CENTER_CODES, user.getCenterCodes(), entry.getValue()));
                    user.setCenterCodes(CastUtils.toList(entry.getValue()));
                    break;
                case "autoProvisioningEnabled":
                    logbooks.add(
                        new EventDiffDto(
                            UserConverter.AUTO_PROVISIONING_ENABLED_KEY,
                            user.isAutoProvisioningEnabled(),
                            entry.getValue()
                        )
                    );
                    user.setAutoProvisioningEnabled(CastUtils.toBoolean(entry.getValue()));
                    break;
                default:
                    throw new IllegalArgumentException(
                        "Unable to patch group " + user.getId() + ": key " + entry.getKey() + " is not allowed"
                    );
            }
        }
        iamLogbookService.updateUserEvent(user, logbooks);
    }

    public void updateOtpForUsersByCustomerId(final boolean otp, final String id) {
        final Query query = new Query(Criteria.where("customerId").is(id));
        final List<User> users = userRepository.findAll(query);
        for (final User u : users) {
            final EventDiffDto evData = new EventDiffDto(UserConverter.OTP_KEY, u.isOtp(), otp);
            u.setOtp(otp);
            userRepository.save(u);
            iamLogbookService.updateUserEvent(u, Arrays.asList(evData));
        }
        final Update update = Update.update("otp", otp);
        userRepository.updateMulti(query, update);
    }

    private User find(final String id, final String customerId, final String message) {
        Assert.isTrue(StringUtils.isNotEmpty(id), message + ": no id");

        // We enforce session customerId (no cross customer allowed for user
        // We make exception for cas user to be allowed for updating all users during provisioning process
        if (!securityService.hasRole(ServicesData.ROLE_PROVISIONING_USER)) {
            Assert.isTrue(
                StringUtils.equals(customerId, getSecurityService().getCustomerId()),
                message + ": customerId " + customerId + " is not allowed"
            );
        }
        return getRepository()
            .findByIdAndCustomerId(id, customerId)
            .orElseThrow(
                () ->
                    new IllegalArgumentException(
                        message + ": no user found for id " + id + " - customerId " + customerId
                    )
            );
    }

    private void checkIsReadonly(final boolean readonly, final String message) {
        Assert.isTrue(!readonly, message + ": readonly user ");
    }

    private void checkGroup(final GroupDto groupDto, final String customerId, final String message) {
        Assert.isTrue(groupDto != null, message + ": group does not exist");

        Assert.isTrue(
            StringUtils.equals(groupDto.getCustomerId(), customerId),
            message + ": group and user customerId must be equals"
        );

        Assert.isTrue(groupDto.isEnabled(), message + ": group must be enabled");
    }

    private void checkSetReadonly(final boolean readonly, final String message) {
        Assert.isTrue(!readonly, message + ": readonly must be set to false");
    }

    private void checkEmail(final String email, final String customerId, final String message) {
        Assert.notNull(email, "email : " + email + " format is not allowed");
        Assert.isTrue(
            Pattern.matches(IamUtils.EMAIL_VALID_REGEXP, email),
            "email : " + email + " format is not allowed"
        );
        Assert.isNull(
            getRepository().findByEmailIgnoreCaseAndCustomerId(email, customerId),
            message + ": mail already exists"
        );
        if (email.matches(ADMIN_EMAIL_PATTERN + ".*")) {
            final Query query = new Query();
            query.addCriteria(Criteria.where("email").regex("^" + ADMIN_EMAIL_PATTERN));
            query.addCriteria(Criteria.where("customerId").is(customerId));
            final Optional<User> adminUser = getRepository().findOne(query);
            Assert.isTrue(adminUser.isEmpty(), message + ": admin user already exists");
        }
    }

    private void checkPhoneNumber(final String phoneNumber) {
        Assert.isTrue(
            Pattern.matches(ApiIamConstants.PHONE_NUMBER_VALID_REGEXP, phoneNumber),
            "Phone Number : " + phoneNumber + " format is not allowed"
        );
    }

    private void checkLevel(final String level, final String message) {
        Assert.isTrue(
            Pattern.matches(ApiIamConstants.LEVEL_VALID_REGEXP, level),
            "level : " + level + " format is not allowed"
        );
        Assert.isTrue(securityService.isLevelAllowed(level), message + ": level " + level + " is not allowed");
    }

    private void checkGroupId(final String groupId, final String message) {
        Assert.isTrue(groupId != null, message + ": groupId must not be null");
    }

    private void checkCustomer(final String customerId, final String message) {
        final Optional<Customer> customer = customerRepository.findById(customerId);
        Assert.isTrue(customer.isPresent(), message + ": customer does not exist");

        Assert.isTrue(customer.get().isEnabled(), message + ": customer must be enabled");
    }

    private void checkOtp(final UserDto userDto) {
        if (UserTypeEnum.GENERIC == userDto.getType() && !userDto.isOtp()) {
            return;
        }
        checkOtp(userDto.isOtp(), userDto.getEmail(), userDto.getMobile(), userDto.getCustomerId());
    }

    private void checkOtp(final User user, final Boolean userHasOtp) {
        checkOtp(userHasOtp, user.getEmail(), user.getMobile(), user.getCustomerId());
    }

    private void checkOtp(
        final boolean userHasOtp,
        final String userEmail,
        final String userMobile,
        final String customerId
    ) {
        final Customer customer = customerRepository
            .findById(customerId)
            .orElseThrow(
                () ->
                    new ApplicationServerException(
                        "Unable to check opt for user " + userEmail + " - Customer  not found: " + customerId
                    )
            );

        if (OtpEnum.MANDATORY.equals(customer.getOtp()) && !userHasOtp) {
            throw new IllegalArgumentException(
                "Unable to disable otp for user:" + userEmail + " . Otp is mandatory this customer"
            );
        }

        if (OtpEnum.DISABLED.equals(customer.getOtp()) && userHasOtp) {
            throw new IllegalArgumentException(
                "Unable to enable otp for user:" + userEmail + " . Otp is mandatory this customer"
            );
        }

        if (userHasOtp && StringUtils.isEmpty(userMobile)) {
            throw new IllegalArgumentException(
                "Unable to enable otp for user:" + userEmail + " without a mobile phone"
            );
        }
    }

    private GroupDto getGroupDtoByIdByPassSecurity(final String groupId, final String message) {
        try {
            return groupService.getOneByPassSecurity(groupId, Optional.empty());
        } catch (final NotFoundException e) {
            throw new IllegalArgumentException(message + ": group does not exist");
        }
    }

    private GroupDto getGroupDtoById(final String groupId, final String message) {
        try {
            return groupService.getOne(groupId, Optional.empty(), Optional.empty());
        } catch (final NotFoundException e) {
            throw new IllegalArgumentException(message + ": group does not exist");
        }
    }

    public UserDto getDefaultAdminUser(final String customerId) {
        final Optional<Customer> customer = customerRepository.findById(customerId);
        if (customer.isEmpty()) {
            throw new NotFoundException("No customer found for: " + customerId);
        }
        final String email =
            ApiIamConstants.ADMIN_CLIENT_PREFIX_EMAIL +
            CommonConstants.EMAIL_SEPARATOR +
            customer.get().getDefaultEmailDomain().replace(".*", "");

        final ArrayList<CriteriaDefinition> criteria = new ArrayList<>();
        criteria.add(Criteria.where("customerId").in(customerId));
        criteria.add(Criteria.where("email").is(email));

        final List<User> users = getRepository().findAll(criteria);
        if (users.isEmpty()) {
            throw new NotFoundException("No admin user found for email: " + email);
        }
        return convertFromEntityToDto(users.get(0));
    }

    public long countByGroupId(final String profileGroupId) {
        return getRepository().countByGroupId(profileGroupId);
    }

    public void addBasicCustomerAndProofTenantIdentifierInformation(final AuthUserDto userDto) {
        final String id = userDto.getId();
        final Customer customer = customerRepository
            .findById(userDto.getCustomerId())
            .orElseThrow(
                () -> new NotFoundException("Cannot find customer : " + userDto.getCustomerId() + " of the user: " + id)
            );
        userDto.setCustomerIdentifier(customer.getIdentifier());
        final BasicCustomerDto basicCustomerDto = new BasicCustomerDto();
        basicCustomerDto.setId(customer.getId());
        basicCustomerDto.setIdentifier(customer.getIdentifier());
        basicCustomerDto.setName(customer.getName());
        basicCustomerDto.setCode(customer.getCode());
        basicCustomerDto.setCompanyName(customer.getCompanyName());
        basicCustomerDto.setPortalMessages(customer.getPortalMessages());
        basicCustomerDto.setPortalTitles(customer.getPortalTitles());
        if (customer.getGraphicIdentity() != null) {
            final GraphicIdentityDto graphicIdentity = new GraphicIdentityDto();
            graphicIdentity.setHasCustomGraphicIdentity(customer.getGraphicIdentity().isHasCustomGraphicIdentity());
            graphicIdentity.setHeaderDataBase64(customer.getGraphicIdentity().getLogoHeaderBase64());
            graphicIdentity.setFooterDataBase64(customer.getGraphicIdentity().getLogoFooterBase64());
            graphicIdentity.setPortalDataBase64(customer.getGraphicIdentity().getLogoPortalBase64());
            graphicIdentity.setThemeColors(customer.getGraphicIdentity().getThemeColors());
            basicCustomerDto.setGraphicIdentity(graphicIdentity);
        }
        userDto.setBasicCustomer(basicCustomerDto);
        userDto.setProofTenantIdentifier(findTenantByCustomerId(customer.getId(), userDto.getId()).getIdentifier());
    }

    private Tenant findTenantByCustomerId(final String customerId, final String userId) {
        final Optional<Tenant> proofTenant = tenantRepository
            .findByCustomerId(customerId)
            .stream()
            .filter(Tenant::isProof)
            .findFirst();
        if (proofTenant.isEmpty()) {
            throw new NotFoundException(
                "Cannot find any proof tenant attached for customer : " + customerId + " of the user: " + userId
            );
        }

        return proofTenant.get();
    }

    public AuthUserDto loadGroupAndProfiles(final UserDto userDto) {
        final AuthUserDto authUserDto = new AuthUserDto(userDto);

        final String groupId = userDto.getGroupId();
        final GroupDto groupDto = getGroupDtoByIdByPassSecurity(groupId, "Unable to embed group");
        authUserDto.setProfileGroup(groupDto);

        final List<String> profileIds = groupDto.getProfileIds();
        final List<ProfileDto> profiles = profileService.getMany(profileIds.toArray(new String[0]));
        if (profiles.size() != profileIds.size()) {
            final List<String> profilesNotFound = profileIds
                .stream()
                .filter(
                    profileId -> profiles.stream().filter(profile -> profile.getId().equals(profileId)).count() == 0
                )
                .collect(Collectors.toList());

            LOGGER.info("profile non trouvé {} ", profilesNotFound);
            LOGGER.info("profile touvé {}", profileIds);
            throw new ApplicationServerException(
                "Unable to embed group " +
                groupId +
                " for user " +
                userDto.getId() +
                " : one of the profiles does not exist"
            );
        }
        groupDto.setProfiles(profiles);

        profiles.forEach(p -> {
            final Tenant tenant = tenantRepository.findByIdentifier(p.getTenantIdentifier());
            p.setTenantName(tenant.getName());
        });
        return authUserDto;
    }

    @Override
    public void addDataAccessRestrictions(final Collection<CriteriaDefinition> criteria) {
        super.addDataAccessRestrictions(criteria);
    }

    public void addTenantsByAppInformation(final AuthUserDto authUserDto) {
        final Map<String, Set<TenantDto>> tenantsByApp = new HashMap<>();
        if (authUserDto.getProfileGroup().getProfiles() != null) {
            authUserDto
                .getProfileGroup()
                .getProfiles()
                .stream()
                .filter(profile -> profile.getTenantName() != null)
                .forEach(profile -> {
                    if (!tenantsByApp.containsKey(profile.getApplicationName())) {
                        tenantsByApp.put(profile.getApplicationName(), new HashSet<>());
                    }
                    final TenantDto tenant = VitamUIUtils.copyProperties(
                        tenantRepository.findByIdentifier(profile.getTenantIdentifier()),
                        new TenantDto()
                    );
                    tenantsByApp.get(profile.getApplicationName()).add(tenant);
                });
        }

        final List<TenantInformationDto> tenantsData = new ArrayList<>();
        tenantsByApp
            .entrySet()
            .stream()
            .forEach(entry -> {
                final TenantInformationDto appInformations = new TenantInformationDto();
                appInformations.setName(entry.getKey());
                appInformations.setTenants(entry.getValue());
                tenantsData.add(appInformations);
            });

        authUserDto.setTenantsByApp(tenantsData);
    }

    @Override
    public boolean checkExist(final String criteriaJsonString) {
        return super.checkExist(criteriaJsonString);
    }

    @Override
    protected Class<User> getEntityClass() {
        return User.class;
    }

    @Override
    protected UserRepository getRepository() {
        return userRepository;
    }

    @Override
    protected Converter<UserDto, User> getConverter() {
        return userConverter;
    }

    public LogbookOperationsCommonResponseDto findHistoryById(final String id) throws VitamClientException {
        LOGGER.debug("findHistoryById for id " + id);
        final Integer tenantIdentifier = securityService.getTenantIdentifier();
        final VitamContext vitamContext = new VitamContext(tenantIdentifier)
            .setAccessContract(securityService.getTenant(tenantIdentifier).getAccessContractLogbookIdentifier())
            .setApplicationSessionId(securityService.getApplicationId());

        final Optional<User> user = getRepository().findById(id);
        user.orElseThrow(() -> new NotFoundException("No user found with id : %s".formatted(id)));
        final JsonNode body = logbookService
            .findEventsByIdentifierAndCollectionNames(
                user.get().getIdentifier(),
                MongoDbCollections.USERS,
                vitamContext
            )
            .toJsonNode();
        try {
            return JsonUtils.treeToValue(body, LogbookOperationsCommonResponseDto.class, false);
        } catch (final JsonProcessingException e) {
            throw new InternalServerException(VitamRestUtils.PARSING_ERROR_MSG, e);
        }
    }

    /**
     * Get levels matching the given criteria.
     *
     * @param criteriaJsonString criteria as json string
     * @return Matching levels
     */
    public List<String> getLevels(final Optional<String> criteriaJsonString) {
        final Document document = groupFields(criteriaJsonString, CommonConstants.LEVEL_ATTRIBUTE);
        LOGGER.debug("getLevels : {}", document);
        if (document == null) {
            return new ArrayList<>();
        }
        return (List<String>) document.get(CommonConstants.LEVEL_ATTRIBUTE);
    }

    public UserDto patchAnalytics(final Map<String, Object> partialDto) {
        checkAnalyticsAllowedFields(partialDto);

        final String userId;

        if (partialDto.containsKey(USER_ID_ATTRIBUTE)) {
            userId = partialDto.get(USER_ID_ATTRIBUTE).toString();
        } else {
            final AuthUserDto loggedUser = getMe();
            userId = loggedUser.getId();
        }

        final User user = getUserById(userId);

        partialDto.forEach((key, value) -> {
            switch (key) {
                case USER_ID_ATTRIBUTE:
                    break;
                case APPLICATION_ID:
                    patchApplicationAnalytics(user, CastUtils.toString(value));
                    break;
                case "lastTenantIdentifier":
                    patchLastTenantIdentifier(user, CastUtils.toInteger(value));
                    break;
                case "alerts":
                    final ObjectMapper objectMapper = new ObjectMapper();
                    final List<AlertAnalytics> alertAnalytics = objectMapper.convertValue(
                        CastUtils.toList(value),
                        new TypeReference<>() {}
                    );
                    patchAlertsAnalytics(user, alertAnalytics);
                    break;
            }
        });

        return userConverter.convertEntityToDto(getRepository().save(user));
    }

    private User getUserById(final String id) {
        return getRepository()
            .findById(id)
            .orElseThrow(() -> new NotFoundException("No user found with id : %s".formatted(id)));
    }

    private void checkAnalyticsAllowedFields(final Map<String, Object> partialDto) {
        final Set<String> analyticsPatchAllowedFields = Set.of(
            APPLICATION_ID,
            "lastTenantIdentifier",
            "alerts",
            USER_ID_ATTRIBUTE
        );

        if (MapUtils.isEmpty(partialDto)) {
            throw new IllegalArgumentException("Unable to patch user analytics : payload is empty");
        }

        partialDto
            .keySet()
            .forEach(key -> {
                if (!analyticsPatchAllowedFields.contains(key)) {
                    throw new IllegalArgumentException(
                        "Unable to patch user analytics key : %s is not allowed".formatted(key)
                    );
                }
            });
    }

    private void patchApplicationAnalytics(final User user, final String applicationId) {
        checkApplicationAccessPermission(applicationId);
        user.getAnalytics().tagApplicationAsLastUsed(applicationId);
    }

    private void patchLastTenantIdentifier(final User user, final Integer tenantIdentifier) {
        user.getAnalytics().setLastTenantIdentifier(tenantIdentifier);
    }

    private void patchAlertsAnalytics(final User user, final List<AlertAnalytics> alerts) {
        user.getAnalytics().setAlerts(alerts);
    }

    private void checkApplicationAccessPermission(final String applicationId) {
        final List<ApplicationDto> loggedUserApplications = applicationService.getAll(
            Optional.empty(),
            Optional.empty()
        );
        final boolean userHasPermission = loggedUserApplications
            .stream()
            .anyMatch(application -> Objects.equals(application.getIdentifier(), applicationId));
        if (!userHasPermission && !applicationId.equals(PORTAL_APP_IDENTIFIER)) {
            throw new IllegalArgumentException(
                "User has no permission to access to the application : %s".formatted(applicationId)
            );
        }
    }

    private Map<String, String> buildUsersInfoLangMap(final List<String> userInfoIds) {
        return userInfoService
            .getMany(userInfoIds)
            .stream()
            .collect(Collectors.toMap(IdDto::getId, UserInfoDto::getLanguage));
    }

    private Map<String, String> buildUsersGroupNamesMap(final List<String> userGroupIds) {
        return groupService.getMany(userGroupIds).stream().collect(Collectors.toMap(IdDto::getId, GroupDto::getName));
    }

    @Override
    protected Document groupFields(final Optional<String> criteriaJsonString, final String... fields) {
        return super.groupFields(criteriaJsonString, fields);
    }

    public List<User> findByCustomerId(String customerId) {
        return this.userRepository.findByCustomerId(customerId);
    }

    @Override
    protected void addRestriction(final String key, final QueryDto query) {
        switch (key) {
            case TYPE_KEY:
                addTypeRestriction(query);
                break;
            case LEVEL_KEY:
                addLevelRestriction(query);
                break;
            default:
                throw new NotImplementedException("Restriction not defined for key: " + key);
        }
    }

    /**
     * If the user is not an admin, he can see only users with a sub LEVEL and himself
     * Example : Users { id: 10, level: ROOT} can see only users with a LEVEL : ROOT..* and himself
     *
     * @param query query
     */
    private void addLevelRestriction(final QueryDto query) {
        final QueryDto levelQuery = new QueryDto();
        levelQuery.setQueryOperator(QueryOperator.OR);
        levelQuery.addCriterion(LEVEL_KEY, securityService.getLevel() + ".", CriterionOperator.STARTWITH);
        levelQuery.addCriterion("id", securityService.getUser().getId(), CriterionOperator.EQUALS);
        query.addQuery(levelQuery);
    }

    private void addTypeRestriction(final QueryDto criteria) {
        final Optional<Criterion> typeCriterion = criteria.find(TYPE_KEY);
        if (typeCriterion.isPresent()) {
            final Criterion criterion = typeCriterion.get();
            final UserTypeEnum userType = EnumUtils.stringToEnum(UserTypeEnum.class, criterion.getValue().toString());
            if (
                !(criterion.getOperator().equals(CriterionOperator.EQUALS) && userType.equals(UserTypeEnum.NOMINATIVE))
            ) {
                throw new ForbiddenException("User's type %s is not allowed".formatted(userType));
            }
        } else {
            criteria.addCriterion(new Criterion(TYPE_KEY, UserTypeEnum.NOMINATIVE, CriterionOperator.EQUALS));
        }
    }

    @Override
    protected Collection<String> getAllowedKeys() {
        return List.of(
            "id",
            "lastname",
            "firstname",
            "identifier",
            "groupId",
            "language",
            "email",
            "otp",
            "subrogeable",
            "phone",
            "mobile",
            "lastConnection",
            "status",
            LEVEL_KEY,
            TYPE_KEY,
            CUSTOMER_ID_KEY,
            "siteCode",
            "centerCodes"
        );
    }

    @Override
    protected Collection<String> getRestrictedKeys() {
        final Collection<String> restrictedKeys = new ArrayList<>(Arrays.asList(CUSTOMER_ID_KEY, TYPE_KEY, LEVEL_KEY));
        if (securityService.hasRole(ServicesData.ROLE_GENERIC_USERS)) {
            restrictedKeys.remove(TYPE_KEY);
        }
        if (securityService.userIsRootLevel()) {
            restrictedKeys.remove(LEVEL_KEY);
        }

        return restrictedKeys;
    }

    @Override
    protected Collection<String> getRestrictedKeys(final QueryDto query) {
        Collection<String> restrictedKeys = getRestrictedKeys();
        if (securityService.hasRole(ServicesData.ROLE_GET_USERS_ALL_CUSTOMERS)) {
            restrictedKeys.remove(CUSTOMER_ID_KEY);
        }
        return restrictedKeys;
    }

    @Override
    protected String getVersionApiCriteria() {
        return CRITERIA_VERSION_V2;
    }
}
