/*
 * Decompiled with CFR 0.152.
 */
package fr.gouv.vitam.common.storage.s3;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Stopwatch;
import fr.gouv.vitam.common.ParametersChecker;
import fr.gouv.vitam.common.VitamConfiguration;
import fr.gouv.vitam.common.alert.AlertService;
import fr.gouv.vitam.common.alert.AlertServiceImpl;
import fr.gouv.vitam.common.digest.DigestType;
import fr.gouv.vitam.common.logging.VitamLogger;
import fr.gouv.vitam.common.logging.VitamLoggerFactory;
import fr.gouv.vitam.common.model.MetadatasObject;
import fr.gouv.vitam.common.model.storage.ObjectEntry;
import fr.gouv.vitam.common.performance.PerformanceLogger;
import fr.gouv.vitam.common.retryable.RetryableOnException;
import fr.gouv.vitam.common.retryable.RetryableParameters;
import fr.gouv.vitam.common.storage.ContainerInformation;
import fr.gouv.vitam.common.storage.StorageConfiguration;
import fr.gouv.vitam.common.storage.cas.container.api.ContentAddressableStorageAbstract;
import fr.gouv.vitam.common.storage.cas.container.api.MetadatasStorageObject;
import fr.gouv.vitam.common.storage.cas.container.api.ObjectContent;
import fr.gouv.vitam.common.storage.cas.container.api.ObjectListingListener;
import fr.gouv.vitam.common.storage.constants.ErrorMessage;
import fr.gouv.vitam.common.storage.s3.BucketNameUtils;
import fr.gouv.vitam.common.stream.ExactSizeInputStream;
import fr.gouv.vitam.common.stream.StreamUtils;
import fr.gouv.vitam.workspace.api.exception.ContentAddressableStorageDigestMismatchException;
import fr.gouv.vitam.workspace.api.exception.ContentAddressableStorageException;
import fr.gouv.vitam.workspace.api.exception.ContentAddressableStorageNotFoundException;
import fr.gouv.vitam.workspace.api.exception.ContentAddressableStorageServerException;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;
import org.apache.commons.io.input.BoundedInputStream;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.ssl.SSLContextBuilder;
import software.amazon.awssdk.auth.credentials.AwsBasicCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProvider;
import software.amazon.awssdk.auth.credentials.StaticCredentialsProvider;
import software.amazon.awssdk.core.ResponseInputStream;
import software.amazon.awssdk.core.checksums.RequestChecksumCalculation;
import software.amazon.awssdk.core.exception.SdkException;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.http.ContentStreamProvider;
import software.amazon.awssdk.http.SdkHttpClient;
import software.amazon.awssdk.http.apache.ApacheHttpClient;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.retries.StandardRetryStrategy;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.S3ClientBuilder;
import software.amazon.awssdk.services.s3.S3Configuration;
import software.amazon.awssdk.services.s3.model.AbortMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.BucketAlreadyExistsException;
import software.amazon.awssdk.services.s3.model.BucketAlreadyOwnedByYouException;
import software.amazon.awssdk.services.s3.model.CompleteMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.CompletedPart;
import software.amazon.awssdk.services.s3.model.CopyObjectRequest;
import software.amazon.awssdk.services.s3.model.CreateBucketRequest;
import software.amazon.awssdk.services.s3.model.CreateMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.CreateMultipartUploadResponse;
import software.amazon.awssdk.services.s3.model.DeleteObjectRequest;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.GetObjectResponse;
import software.amazon.awssdk.services.s3.model.HeadBucketRequest;
import software.amazon.awssdk.services.s3.model.HeadObjectRequest;
import software.amazon.awssdk.services.s3.model.HeadObjectResponse;
import software.amazon.awssdk.services.s3.model.ListObjectsV2Request;
import software.amazon.awssdk.services.s3.model.ListObjectsV2Response;
import software.amazon.awssdk.services.s3.model.MetadataDirective;
import software.amazon.awssdk.services.s3.model.NoSuchBucketException;
import software.amazon.awssdk.services.s3.model.NoSuchKeyException;
import software.amazon.awssdk.services.s3.model.PutObjectRequest;
import software.amazon.awssdk.services.s3.model.S3Exception;
import software.amazon.awssdk.services.s3.model.S3Object;
import software.amazon.awssdk.services.s3.model.UploadPartCopyRequest;
import software.amazon.awssdk.services.s3.model.UploadPartCopyResponse;
import software.amazon.awssdk.services.s3.model.UploadPartRequest;
import software.amazon.awssdk.services.s3.model.UploadPartResponse;

public class AmazonS3V2
extends ContentAddressableStorageAbstract {
    private static final VitamLogger LOGGER = VitamLoggerFactory.getInstance(AmazonS3V2.class);
    private static final String X_OBJECT_META_DIGEST = "Digest";
    private static final String X_OBJECT_META_DIGEST_TYPE = "Digest-Type";
    private static final String NO_SUCH_BUCKET_ERROR_CODE = "NoSuchBucket";
    private static final AlertService ALERT_SERVICE = new AlertServiceImpl();
    private static final long MB_TO_BYTES = 0x100000L;
    private static final int MAX_RETRIES = 3;
    private static final int NO_RETRIES = 1;
    private final boolean s3DisableMultipartUpload;
    private final long s3MaxUploadPartSize;
    private final int s3MultiPartCleanNbRetries;
    private final int s3MultiPartCleanWaitingTimeInMilliseconds;
    private final SdkHttpClient httpClient;
    private final S3Client s3ClientWithRetry;
    private final S3Client s3ClientWithoutRetry;

    public AmazonS3V2(StorageConfiguration configuration) throws KeyManagementException, NoSuchAlgorithmException, KeyStoreException, CertificateException, IOException {
        super(configuration);
        AmazonS3V2.checkConfiguration(configuration);
        this.s3DisableMultipartUpload = configuration.isS3DisableMultipartUpload();
        this.s3MaxUploadPartSize = configuration.getS3MaxUploadPartSizeMB() * 0x100000L;
        this.s3MultiPartCleanNbRetries = configuration.getS3MultiPartCleanNbRetries();
        this.s3MultiPartCleanWaitingTimeInMilliseconds = configuration.getS3MultiPartCleanWaitingTimeInMilliseconds();
        this.httpClient = AmazonS3V2.createHttpClient(configuration);
        this.s3ClientWithRetry = AmazonS3V2.createS3Client(configuration, this.httpClient, true);
        this.s3ClientWithoutRetry = AmazonS3V2.createS3Client(configuration, this.httpClient, false);
    }

    @VisibleForTesting
    AmazonS3V2(StorageConfiguration configuration, SdkHttpClient httpClient, S3Client s3ClientWithRetry, S3Client s3ClientWithoutRetry) {
        super(configuration);
        this.httpClient = httpClient;
        this.s3ClientWithRetry = s3ClientWithRetry;
        this.s3ClientWithoutRetry = s3ClientWithoutRetry;
        this.s3DisableMultipartUpload = configuration.isS3DisableMultipartUpload();
        this.s3MaxUploadPartSize = configuration.getS3MaxUploadPartSizeMB() * 0x100000L;
        this.s3MultiPartCleanNbRetries = configuration.getS3MultiPartCleanNbRetries();
        this.s3MultiPartCleanWaitingTimeInMilliseconds = configuration.getS3MultiPartCleanWaitingTimeInMilliseconds();
    }

    private static void checkConfiguration(StorageConfiguration configuration) {
        if (configuration.isS3DisableMultipartUpload()) {
            return;
        }
        if (configuration.getS3MaxUploadPartSizeMB() < 5L || configuration.getS3MaxUploadPartSizeMB() > 5120L) {
            throw new IllegalArgumentException("Invalid max part upload size %d. Valid values must be in the range [%d MB - %d MB]".formatted(configuration.getS3MaxUploadPartSizeMB(), 5120, 5120));
        }
    }

    private static SdkHttpClient createHttpClient(StorageConfiguration configuration) throws KeyStoreException, IOException, NoSuchAlgorithmException, CertificateException, KeyManagementException {
        return ApacheHttpClient.builder().maxConnections(Integer.valueOf(configuration.getS3MaxConnections())).connectionTimeout(Duration.ofMillis(configuration.getS3ConnectionTimeout())).socketTimeout(Duration.ofMillis(configuration.getS3SocketTimeout())).socketFactory((ConnectionSocketFactory)AmazonS3V2.createSSLConnectionSocketFactory(configuration)).build();
    }

    private static SSLConnectionSocketFactory createSSLConnectionSocketFactory(StorageConfiguration configuration) throws KeyStoreException, IOException, NoSuchAlgorithmException, CertificateException, KeyManagementException {
        if (!configuration.getS3Endpoint().startsWith("https://")) {
            return null;
        }
        KeyStore trustStore = KeyStore.getInstance(KeyStore.getDefaultType());
        try (FileInputStream fis = new FileInputStream(configuration.getS3TrustStore());){
            trustStore.load(fis, configuration.getS3TrustStorePassword().toCharArray());
        }
        SSLContext sslContext = SSLContextBuilder.create().loadTrustMaterial(trustStore, null).build();
        Object hostnameVerifier = configuration.isS3IgnoreCertificateHostnameValidation() ? NoopHostnameVerifier.INSTANCE : SSLConnectionSocketFactory.getDefaultHostnameVerifier();
        return new SSLConnectionSocketFactory(sslContext, (HostnameVerifier)hostnameVerifier);
    }

    private static S3Client createS3Client(StorageConfiguration configuration, SdkHttpClient httpClient, boolean retryable) {
        AwsBasicCredentials credentials = AwsBasicCredentials.create((String)configuration.getS3AccessKey(), (String)configuration.getS3SecretKey());
        S3Configuration serviceConfiguration = (S3Configuration)S3Configuration.builder().pathStyleAccessEnabled(Boolean.valueOf(configuration.isS3PathStyleAccessEnabled())).build();
        return (S3Client)((S3ClientBuilder)((S3ClientBuilder)((S3ClientBuilder)((S3ClientBuilder)((S3ClientBuilder)((S3ClientBuilder)((S3ClientBuilder)S3Client.builder().endpointOverride(URI.create(configuration.getS3Endpoint()))).requestChecksumCalculation(RequestChecksumCalculation.WHEN_REQUIRED)).region(StringUtils.isEmpty((CharSequence)configuration.getS3RegionName()) ? Region.US_EAST_1 : Region.of((String)configuration.getS3RegionName()))).credentialsProvider((AwsCredentialsProvider)StaticCredentialsProvider.create((AwsCredentials)credentials))).httpClient(httpClient)).serviceConfiguration(serviceConfiguration)).overrideConfiguration(cfg -> cfg.apiCallTimeout(configuration.getS3ClientExecutionTimeout() != 0 ? Duration.ofMillis(configuration.getS3ClientExecutionTimeout()) : null).apiCallAttemptTimeout(configuration.getS3RequestTimeout() != 0 ? Duration.ofMillis(configuration.getS3RequestTimeout()) : null).retryStrategy((RetryStrategy)((StandardRetryStrategy.Builder)StandardRetryStrategy.builder().maxAttempts(retryable ? 3 : 1)).build()))).build();
    }

    @Override
    public void createContainer(String containerName) throws ContentAddressableStorageServerException {
        LOGGER.debug(String.format("Create container %s", containerName));
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_NAME_IS_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName});
        String bucketName = this.generateBucketName(containerName);
        try {
            CreateBucketRequest request = (CreateBucketRequest)CreateBucketRequest.builder().bucket(bucketName).build();
            this.s3ClientWithRetry.createBucket(request);
        }
        catch (BucketAlreadyExistsException | BucketAlreadyOwnedByYouException e) {
            LOGGER.warn("Container " + containerName + " already exists", e);
        }
        catch (SdkException e) {
            throw new ContentAddressableStorageServerException("Error when trying to create container", (Throwable)e);
        }
    }

    @Override
    public boolean isExistingContainer(String containerName) throws ContentAddressableStorageServerException {
        LOGGER.debug(String.format("Check existence of container %s", containerName));
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_NAME_IS_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName});
        String bucketName = this.generateBucketName(containerName);
        if (super.isExistingContainerInCache(containerName)) {
            return true;
        }
        try {
            boolean exists = this.bucketExists(bucketName);
            this.cacheExistsContainer(containerName, exists);
            return exists;
        }
        catch (SdkException e) {
            throw new ContentAddressableStorageServerException("Error when trying to check existence of container", (Throwable)e);
        }
    }

    private boolean bucketExists(String bucketName) throws SdkException {
        try {
            HeadBucketRequest request = (HeadBucketRequest)HeadBucketRequest.builder().bucket(bucketName).build();
            this.s3ClientWithRetry.headBucket(request);
            return true;
        }
        catch (NoSuchBucketException e) {
            LOGGER.debug("Container '" + bucketName + " does not exist", (Throwable)e);
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void writeObject(String containerName, String objectName, InputStream inputStream, DigestType digestType, long size) throws ContentAddressableStorageException {
        LOGGER.debug(String.format("Upload object %s in container %s", objectName, containerName));
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_OBJECT_NAMES_ARE_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName, objectName});
        String bucketName = this.generateBucketName(containerName);
        Stopwatch times = Stopwatch.createStarted();
        try {
            if (this.useMultipartUpload(size)) {
                this.putLargeObject(bucketName, objectName, inputStream, size);
            } else {
                this.putSmallObject(bucketName, objectName, inputStream, size);
            }
        }
        finally {
            PerformanceLogger.getInstance().log("STP_Offer_" + this.getConfiguration().getProvider(), containerName, "REAL_S3_PUT_OBJECT", times.elapsed(TimeUnit.MILLISECONDS));
        }
    }

    @Override
    public void checkObjectDigestAndStoreDigest(String containerName, String objectName, String objectDigest, DigestType digestType, long size) throws ContentAddressableStorageException {
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_OBJECT_NAMES_ARE_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName, objectName});
        String bucketName = this.generateBucketName(containerName);
        String computedDigest = this.computeObjectDigest(containerName, objectName, digestType);
        if (!objectDigest.equals(computedDigest)) {
            throw new ContentAddressableStorageDigestMismatchException("Illegal state for container " + containerName + " and object " + objectName + ". Stream digest " + objectDigest + " is not equal to computed digest " + computedDigest);
        }
        this.storeDigest(containerName, objectName, digestType, objectDigest, bucketName, size);
    }

    private void putLargeObject(String bucketName, String objectName, InputStream stream, long size) throws ContentAddressableStorageServerException {
        LOGGER.info("Uploading large object {}/{} ({} bytes)", new Object[]{bucketName, objectName, size});
        String uploadId = null;
        boolean uploadSucceeded = false;
        try {
            Stopwatch stopwatch = Stopwatch.createStarted();
            uploadId = this.initiateMultipartUpload(bucketName, objectName, Collections.emptyMap());
            ArrayList<CompletedPart> completedParts = new ArrayList<CompletedPart>();
            long sentBytes = 0L;
            long nbParts = (size + this.s3MaxUploadPartSize - 1L) / this.s3MaxUploadPartSize;
            int partNumber = 1;
            while (sentBytes < size) {
                long partSize = Math.min(this.s3MaxUploadPartSize, size - sentBytes);
                CompletedPart completedPart = this.uploadPart(bucketName, objectName, stream, partNumber, nbParts, partSize, uploadId);
                completedParts.add(completedPart);
                sentBytes += partSize;
                ++partNumber;
            }
            this.completeMultipartUpload(bucketName, objectName, uploadId, completedParts);
            uploadSucceeded = true;
            LOGGER.info("Large object {}/{} uploaded successfully  ({} bytes, {} ms)", new Object[]{bucketName, objectName, size, stopwatch.elapsed(TimeUnit.MILLISECONDS)});
        }
        catch (SdkException e) {
            throw new ContentAddressableStorageServerException("Error when trying to upload object " + bucketName + "/" + objectName, (Throwable)e);
        }
        finally {
            StreamUtils.closeSilently((InputStream)stream);
            if (uploadId != null && !uploadSucceeded) {
                this.tryCleanupMultiPartUpload(bucketName, objectName, uploadId);
            }
        }
    }

    private String initiateMultipartUpload(String containerName, String objectName, Map<String, String> metadata) throws SdkException {
        LOGGER.debug("Initiating multipart upload {}/{}", (Object)containerName, (Object)objectName);
        CreateMultipartUploadRequest request = (CreateMultipartUploadRequest)CreateMultipartUploadRequest.builder().bucket(containerName).key(objectName).metadata(metadata).build();
        CreateMultipartUploadResponse multipartUpload = this.s3ClientWithRetry.createMultipartUpload(request);
        String uploadId = multipartUpload.uploadId();
        LOGGER.debug("Initiated multipart upload {}", (Object)uploadId);
        return uploadId;
    }

    private CompletedPart uploadPart(String containerName, String objectName, InputStream stream, int partNumber, long nbParts, long partSize, String uploadId) throws ContentAddressableStorageServerException {
        CompletedPart completedPart;
        block8: {
            LOGGER.info("Multipart upload of {}/{} - Part {}/{} ({} bytes)", new Object[]{containerName, objectName, partNumber, nbParts, partSize});
            BoundedInputStream partInputStream = ((BoundedInputStream.Builder)((BoundedInputStream.Builder)((BoundedInputStream.Builder)BoundedInputStream.builder().setInputStream(stream)).setMaxCount(partSize)).setPropagateClose(false)).get();
            try {
                UploadPartRequest request = (UploadPartRequest)UploadPartRequest.builder().bucket(containerName).uploadId(uploadId).contentLength(Long.valueOf(partSize)).partNumber(Integer.valueOf(partNumber)).key(objectName).build();
                UploadPartResponse uploadPartResult = this.s3ClientWithoutRetry.uploadPart(request, AmazonS3V2.fixedLengthInputStreamRequestBody((InputStream)partInputStream, partSize));
                completedPart = (CompletedPart)CompletedPart.builder().partNumber(Integer.valueOf(partNumber)).eTag(uploadPartResult.eTag()).build();
                if (partInputStream == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (partInputStream != null) {
                        try {
                            partInputStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new ContentAddressableStorageServerException((Throwable)e);
                }
            }
            partInputStream.close();
        }
        return completedPart;
    }

    private void completeMultipartUpload(String bucketName, String objectName, String uploadId, List<CompletedPart> completedParts) {
        LOGGER.debug("Completing multi-part upload {} of {}/{}", new Object[]{uploadId, bucketName, objectName});
        CompleteMultipartUploadRequest request = (CompleteMultipartUploadRequest)CompleteMultipartUploadRequest.builder().bucket(bucketName).key(objectName).uploadId(uploadId).multipartUpload(u -> u.parts((Collection)completedParts)).build();
        this.s3ClientWithRetry.completeMultipartUpload(request);
    }

    private void tryCleanupMultiPartUpload(String bucketName, String objectName, String uploadId) {
        try {
            new RetryableOnException(new RetryableParameters(this.s3MultiPartCleanNbRetries, this.s3MultiPartCleanWaitingTimeInMilliseconds, this.s3MultiPartCleanWaitingTimeInMilliseconds, this.s3MultiPartCleanWaitingTimeInMilliseconds, TimeUnit.MILLISECONDS)).exec(() -> {
                LOGGER.error("Multi-part upload for object {}/{} with id {} failed. Cleaning up...", new Object[]{bucketName, objectName, uploadId});
                AbortMultipartUploadRequest request = (AbortMultipartUploadRequest)AbortMultipartUploadRequest.builder().bucket(bucketName).key(objectName).uploadId(uploadId).build();
                this.s3ClientWithRetry.abortMultipartUpload(request);
                return null;
            });
            LOGGER.warn("Cleanup of multi-part upload for object {}/{} with id {} succeeded", new Object[]{bucketName, objectName, uploadId});
        }
        catch (Exception e) {
            String msg = String.format("An error occurred during multi-part upload for object %s/%s with id %s.Please cleanup your S3 storage server manually using the S3 AbortMultipartUpload API", bucketName, objectName, uploadId);
            ALERT_SERVICE.createAlert(msg);
            LOGGER.error(msg, (Throwable)e);
        }
    }

    private void putSmallObject(String bucketName, String objectName, InputStream stream, long size) throws ContentAddressableStorageServerException, ContentAddressableStorageNotFoundException {
        try {
            PutObjectRequest request = (PutObjectRequest)PutObjectRequest.builder().bucket(bucketName).key(objectName).contentLength(Long.valueOf(size)).build();
            this.s3ClientWithoutRetry.putObject(request, AmazonS3V2.fixedLengthInputStreamRequestBody(stream, size));
        }
        catch (S3Exception e) {
            if (NO_SUCH_BUCKET_ERROR_CODE.equals(e.awsErrorDetails().errorCode())) {
                throw new ContentAddressableStorageNotFoundException("Error when trying to upload object : container does not exists", (Throwable)e);
            }
            throw new ContentAddressableStorageServerException("Error when trying to upload object", (Throwable)e);
        }
        catch (IOException | SdkException e) {
            throw new ContentAddressableStorageServerException("Error when trying to upload object", e);
        }
    }

    private void storeDigest(String containerName, String objectName, DigestType digestType, String digest, String bucketName, long size) throws ContentAddressableStorageException {
        Stopwatch stopwatch = Stopwatch.createStarted();
        if (this.useMultipartUpload(size)) {
            this.storeLargeObjectDigest(bucketName, objectName, digestType, digest, size);
        } else {
            this.storeSmallObjectDigest(bucketName, objectName, digestType, digest);
        }
        PerformanceLogger.getInstance().log("STP_Offer_" + this.getConfiguration().getProvider(), containerName, "STORE_DIGEST_IN_METADATA", stopwatch.elapsed(TimeUnit.MILLISECONDS));
    }

    private void storeSmallObjectDigest(String bucketName, String objectName, DigestType digestType, String digest) throws ContentAddressableStorageException {
        Map<String, String> metadataToUpdate = AmazonS3V2.createObjectMetadata(digestType, digest);
        try {
            CopyObjectRequest request = (CopyObjectRequest)CopyObjectRequest.builder().sourceBucket(bucketName).sourceKey(objectName).destinationBucket(bucketName).destinationKey(objectName).metadata(metadataToUpdate).metadataDirective(MetadataDirective.REPLACE).build();
            this.s3ClientWithRetry.copyObject(request);
        }
        catch (SdkException e) {
            throw new ContentAddressableStorageServerException("Error when trying to update metadata of object", (Throwable)e);
        }
    }

    private static Map<String, String> createObjectMetadata(DigestType digestType, String digest) {
        return Map.ofEntries(Map.entry(X_OBJECT_META_DIGEST, digest), Map.entry(X_OBJECT_META_DIGEST_TYPE, digestType.getName()));
    }

    private void storeLargeObjectDigest(String bucketName, String objectName, DigestType digestType, String digest, long size) throws ContentAddressableStorageException {
        LOGGER.info("Updating digest for large object {}/{} (digest: {}, size: {} bytes)", new Object[]{bucketName, objectName, digest, size});
        String copyObjectUploadId = null;
        boolean copySucceeded = false;
        try {
            Stopwatch stopwatch = Stopwatch.createStarted();
            Map<String, String> metadataToUpdate = AmazonS3V2.createObjectMetadata(digestType, digest);
            copyObjectUploadId = this.initiateMultipartUpload(bucketName, objectName, metadataToUpdate);
            ArrayList<CompletedPart> completedParts = new ArrayList<CompletedPart>();
            long offset = 0L;
            long nbParts = (size + this.s3MaxUploadPartSize - 1L) / this.s3MaxUploadPartSize;
            int partNumber = 1;
            while (offset < size) {
                long partSize = Math.min(this.s3MaxUploadPartSize, size - offset);
                CompletedPart completedPart = this.copyPart(bucketName, objectName, partNumber, nbParts, partSize, copyObjectUploadId, offset);
                completedParts.add(completedPart);
                offset += partSize;
                ++partNumber;
            }
            this.completeMultipartUpload(bucketName, objectName, copyObjectUploadId, completedParts);
            copySucceeded = true;
            LOGGER.info("Large object {}/{} updated successfully with digest {} ({} bytes, {} ms)", new Object[]{bucketName, objectName, digest, size, stopwatch.elapsed(TimeUnit.MILLISECONDS)});
            if (copyObjectUploadId != null && !copySucceeded) {
                this.tryCleanupMultiPartUpload(bucketName, objectName, copyObjectUploadId);
            }
        }
        catch (SdkException e) {
            try {
                throw new ContentAddressableStorageServerException("Error when trying to updating large object digest " + bucketName + "/" + objectName, (Throwable)e);
            }
            catch (Throwable throwable) {
                if (copyObjectUploadId != null && !copySucceeded) {
                    this.tryCleanupMultiPartUpload(bucketName, objectName, copyObjectUploadId);
                }
                throw throwable;
            }
        }
    }

    private CompletedPart copyPart(String containerName, String objectName, int partNumber, long nbParts, long partSize, String uploadId, long offset) {
        LOGGER.info("Copying multi-part for digest update of {}/{} - Part {}/{} ({} bytes)", new Object[]{containerName, objectName, partNumber, nbParts, partSize});
        UploadPartCopyRequest uploadPartCopyRequest = (UploadPartCopyRequest)UploadPartCopyRequest.builder().sourceBucket(containerName).sourceKey(objectName).destinationBucket(containerName).destinationKey(objectName).uploadId(uploadId).copySourceRange(String.format("bytes=%d-%d", offset, offset + partSize - 1L)).partNumber(Integer.valueOf(partNumber)).build();
        UploadPartCopyResponse uploadPartCopyResponse = this.s3ClientWithRetry.uploadPartCopy(uploadPartCopyRequest);
        return (CompletedPart)CompletedPart.builder().partNumber(Integer.valueOf(partNumber)).eTag(uploadPartCopyResponse.copyPartResult().eTag()).build();
    }

    @Override
    public ObjectContent getObject(String containerName, String objectName) throws ContentAddressableStorageException {
        LOGGER.debug(String.format("Download object %s from container %s", objectName, containerName));
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_OBJECT_NAMES_ARE_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName, objectName});
        String bucketName = this.generateBucketName(containerName);
        try {
            GetObjectRequest request = (GetObjectRequest)GetObjectRequest.builder().bucket(bucketName).key(objectName).build();
            ResponseInputStream responseInputStream = this.s3ClientWithRetry.getObject(request);
            long size = ((GetObjectResponse)responseInputStream.response()).contentLength();
            return new ObjectContent((InputStream)responseInputStream, size);
        }
        catch (NoSuchKeyException e) {
            throw new ContentAddressableStorageNotFoundException("Error when trying to download object " + containerName + "/" + objectName + ". Object not found.", (Throwable)e);
        }
        catch (NoSuchBucketException e) {
            throw new ContentAddressableStorageNotFoundException("Error when trying to download object " + containerName + "/" + objectName + ". Container not found.", (Throwable)e);
        }
        catch (SdkException e) {
            throw new ContentAddressableStorageServerException("Error when trying to download object " + containerName + "/" + objectName, (Throwable)e);
        }
    }

    @Override
    public void deleteObject(String containerName, String objectName) throws ContentAddressableStorageException {
        LOGGER.debug(String.format("Delete object %s from container %s", objectName, containerName));
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_OBJECT_NAMES_ARE_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName, objectName});
        String bucketName = this.generateBucketName(containerName);
        try {
            DeleteObjectRequest request = (DeleteObjectRequest)DeleteObjectRequest.builder().bucket(bucketName).key(objectName).build();
            this.s3ClientWithRetry.deleteObject(request);
        }
        catch (NoSuchBucketException e) {
            throw new ContentAddressableStorageNotFoundException("Error when trying to delete object " + containerName + "/" + objectName + ". Container not found.", (Throwable)e);
        }
        catch (SdkException e) {
            throw new ContentAddressableStorageServerException("Error when trying to delete object " + containerName + "/" + objectName, (Throwable)e);
        }
    }

    @Override
    public boolean isExistingObject(String containerName, String objectName) throws ContentAddressableStorageServerException {
        LOGGER.debug(String.format("Check existence of object %s in container %s", objectName, containerName));
        String bucketName = this.generateBucketName(containerName);
        try {
            try {
                HeadObjectRequest request = (HeadObjectRequest)HeadObjectRequest.builder().bucket(bucketName).key(objectName).build();
                this.s3ClientWithRetry.headObject(request);
                return true;
            }
            catch (NoSuchBucketException | NoSuchKeyException e) {
                return false;
            }
        }
        catch (SdkException e) {
            throw new ContentAddressableStorageServerException("Error when trying to check existence of object " + containerName + "/" + objectName, (Throwable)e);
        }
    }

    @Override
    public String getObjectDigest(String containerName, String objectName, DigestType digestType, boolean noCache) throws ContentAddressableStorageException {
        LOGGER.debug(String.format("Get digest of object %s in container %s", objectName, containerName));
        if (!noCache) {
            Stopwatch stopwatch = Stopwatch.createStarted();
            String bucketName = this.generateBucketName(containerName);
            try {
                HeadObjectRequest request = (HeadObjectRequest)HeadObjectRequest.builder().bucket(bucketName).key(objectName).build();
                HeadObjectResponse headObjectResponse = this.s3ClientWithRetry.headObject(request);
                PerformanceLogger.getInstance().log("STP_Offer_" + this.getConfiguration().getProvider(), containerName, "READ_DIGEST_FROM_METADATA", stopwatch.elapsed(TimeUnit.MILLISECONDS));
                return this.getDigestFromObjectMetadata(containerName, objectName, digestType, bucketName, headObjectResponse);
            }
            catch (NoSuchKeyException e) {
                throw new ContentAddressableStorageNotFoundException("Error when trying to compute digest of object " + objectName + " from container " + containerName + ". Object not found.", (Throwable)e);
            }
            catch (SdkException e) {
                throw new ContentAddressableStorageServerException("Error when trying to compute digest of object " + containerName + "/" + objectName, (Throwable)e);
            }
        }
        return this.computeObjectDigest(containerName, objectName, digestType);
    }

    private String getDigestFromObjectMetadata(String containerName, String objectName, DigestType digestType, String bucketName, HeadObjectResponse headObjectResponse) throws ContentAddressableStorageException {
        if (null != headObjectResponse && headObjectResponse.metadata().containsKey(X_OBJECT_META_DIGEST) && headObjectResponse.metadata().containsKey(X_OBJECT_META_DIGEST_TYPE) && digestType.getName().equals(headObjectResponse.metadata().get(X_OBJECT_META_DIGEST_TYPE)) && null != headObjectResponse.metadata().get(X_OBJECT_META_DIGEST)) {
            return (String)headObjectResponse.metadata().get(X_OBJECT_META_DIGEST);
        }
        LOGGER.warn(String.format("Could not retrieve cached digest of object '%s' in container '%s'. Recomputing digest", objectName, containerName));
        Pair<String, Long> objectDigestAndSize = this.computeObjectDigestAndSize(containerName, objectName, digestType);
        String digestToStore = (String)objectDigestAndSize.getKey();
        Long objectSize = (Long)objectDigestAndSize.getValue();
        this.storeDigest(containerName, objectName, digestType, digestToStore, bucketName, objectSize);
        return digestToStore;
    }

    @Override
    public ContainerInformation getContainerInformation(String containerName) throws ContentAddressableStorageNotFoundException, ContentAddressableStorageServerException {
        LOGGER.debug(String.format("Get information of container %s", containerName));
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_NAME_IS_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName});
        ContainerInformation containerInformation = new ContainerInformation();
        containerInformation.setUsableSpace(-1L);
        return containerInformation;
    }

    @Override
    public MetadatasObject getObjectMetadata(String containerName, String objectId, boolean noCache) throws ContentAddressableStorageException {
        LOGGER.debug(String.format("Get metadata of object %s in container %s", objectId, containerName));
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_OBJECT_NAMES_ARE_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName, objectId});
        String bucketName = this.generateBucketName(containerName);
        try {
            MetadatasStorageObject result = new MetadatasStorageObject();
            HeadObjectRequest request = (HeadObjectRequest)HeadObjectRequest.builder().bucket(bucketName).key(objectId).build();
            HeadObjectResponse headObjectResponse = this.s3ClientWithRetry.headObject(request);
            result.setType(containerName.split("_")[1]);
            result.setObjectName(objectId);
            result.setDigest(noCache ? this.computeObjectDigest(containerName, objectId, VitamConfiguration.getDefaultDigestType()) : this.getDigestFromObjectMetadata(containerName, objectId, VitamConfiguration.getDefaultDigestType(), bucketName, headObjectResponse));
            result.setFileSize(headObjectResponse.contentLength());
            result.setLastModifiedDate(headObjectResponse.lastModified().toString());
            return result;
        }
        catch (NoSuchKeyException e) {
            throw new ContentAddressableStorageNotFoundException("Error when trying to get metadata of object " + containerName + "/" + objectId + ". Object not found.", (Throwable)e);
        }
        catch (SdkException e) {
            throw new ContentAddressableStorageServerException("Error when trying to get metadata of object " + containerName + "/" + objectId, (Throwable)e);
        }
    }

    @Override
    public void listContainer(String containerName, ObjectListingListener objectListingListener) throws ContentAddressableStorageNotFoundException, ContentAddressableStorageServerException, IOException {
        LOGGER.debug(String.format("Listing of object in container %s", containerName));
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_NAME_IS_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName});
        String bucketName = this.generateBucketName(containerName);
        try {
            ListObjectsV2Response listObjectsV2Response;
            String continuationToken = null;
            do {
                ListObjectsV2Request listObjectsV2Request = (ListObjectsV2Request)ListObjectsV2Request.builder().bucket(bucketName).maxKeys(Integer.valueOf(this.getConfiguration().getS3ListObjectBulkSize())).continuationToken(continuationToken).build();
                listObjectsV2Response = this.s3ClientWithRetry.listObjectsV2(listObjectsV2Request);
                for (S3Object s3Object : listObjectsV2Response.contents()) {
                    LOGGER.debug("Found object {}/{} ({} bytes)", new Object[]{containerName, s3Object.key(), s3Object.size()});
                    objectListingListener.handleObjectEntry(new ObjectEntry(s3Object.key(), s3Object.size().longValue()));
                }
            } while ((continuationToken = listObjectsV2Response.nextContinuationToken()) != null);
        }
        catch (NoSuchBucketException e) {
            throw new ContentAddressableStorageNotFoundException(ErrorMessage.CONTAINER_NOT_FOUND.getMessage() + containerName, (Throwable)e);
        }
        catch (SdkException e) {
            throw new ContentAddressableStorageServerException("Error when trying to list objects of container " + containerName, (Throwable)e);
        }
    }

    public void close() {
        this.s3ClientWithRetry.close();
        this.s3ClientWithoutRetry.close();
        this.httpClient.close();
    }

    public String generateBucketName(String containerName) {
        String bucketName = containerName.replaceAll("[^A-Za-z0-9]", ".").toLowerCase();
        bucketName = StringUtils.strip((String)bucketName, (String)".");
        LOGGER.debug(String.format("Generated bucket name %s from container name %s", bucketName, containerName));
        BucketNameUtils.validateBucketName(bucketName);
        return bucketName;
    }

    private boolean useMultipartUpload(long size) {
        return !this.s3DisableMultipartUpload && size > this.s3MaxUploadPartSize;
    }

    private static RequestBody fixedLengthInputStreamRequestBody(InputStream inputStream, long contentLength) throws IOException {
        ExactSizeInputStream exactSizeInputStream = new ExactSizeInputStream(inputStream, contentLength);
        AtomicBoolean alreadyConsumed = new AtomicBoolean(false);
        ContentStreamProvider contentStreamProvider = () -> {
            if (alreadyConsumed.getAndSet(true)) {
                throw new IllegalStateException("Cannot create a new stream. Already consumed");
            }
            return exactSizeInputStream;
        };
        return RequestBody.fromContentProvider((ContentStreamProvider)contentStreamProvider, (long)contentLength, (String)"application/octet-stream");
    }
}

