/*
 * Decompiled with CFR 0.152.
 */
package fr.gouv.vitam.workspace.common;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import fr.gouv.vitam.common.CommonMediaType;
import fr.gouv.vitam.common.FileUtil;
import fr.gouv.vitam.common.ParametersChecker;
import fr.gouv.vitam.common.client.AbstractMockClient;
import fr.gouv.vitam.common.digest.Digest;
import fr.gouv.vitam.common.digest.DigestType;
import fr.gouv.vitam.common.guid.GUIDFactory;
import fr.gouv.vitam.common.json.JsonHandler;
import fr.gouv.vitam.common.logging.VitamLogger;
import fr.gouv.vitam.common.logging.VitamLoggerFactory;
import fr.gouv.vitam.common.security.IllegalPathException;
import fr.gouv.vitam.common.security.SafeFileChecker;
import fr.gouv.vitam.common.server.application.VitamHttpHeader;
import fr.gouv.vitam.common.storage.ContainerInformation;
import fr.gouv.vitam.common.storage.StorageConfiguration;
import fr.gouv.vitam.common.storage.compress.ArchiveEntryInputStream;
import fr.gouv.vitam.common.storage.compress.VitamArchiveStreamFactory;
import fr.gouv.vitam.common.storage.constants.ErrorMessage;
import fr.gouv.vitam.common.stream.ExactSizeInputStream;
import fr.gouv.vitam.common.stream.StreamUtils;
import fr.gouv.vitam.workspace.api.exception.ContentAddressableStorageAlreadyExistException;
import fr.gouv.vitam.workspace.api.exception.ContentAddressableStorageCompressedFileException;
import fr.gouv.vitam.workspace.api.exception.ContentAddressableStorageException;
import fr.gouv.vitam.workspace.api.exception.ContentAddressableStorageNotFoundException;
import fr.gouv.vitam.workspace.api.exception.ContentAddressableStorageServerException;
import fr.gouv.vitam.workspace.api.exception.ZipFilesNameNotAllowedException;
import fr.gouv.vitam.workspace.api.model.FileParams;
import fr.gouv.vitam.workspace.api.model.TimeToLive;
import fr.gouv.vitam.workspace.common.BulkMoveEntry;
import fr.gouv.vitam.workspace.common.WorkspaceContentAddressableStorage;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.MultivaluedHashMap;
import jakarta.ws.rs.core.Response;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URLEncoder;
import java.nio.channels.Channels;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.ArchiveException;
import org.apache.commons.compress.archivers.ArchiveInputStream;
import org.apache.commons.compress.archivers.ArchiveOutputStream;
import org.apache.commons.compress.archivers.ArchiveStreamFactory;
import org.apache.commons.compress.archivers.zip.Zip64Mode;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream;
import org.apache.commons.compress.utils.BoundedInputStream;
import org.apache.commons.compress.utils.IOUtils;
import org.apache.commons.io.FileUtils;

public class WorkspaceFileSystem
implements WorkspaceContentAddressableStorage {
    private static final VitamLogger LOGGER = VitamLoggerFactory.getInstance(WorkspaceFileSystem.class);
    private static final String LINUX_PATH_SEPARATOR = "/";
    private final Path root;

    public WorkspaceFileSystem(StorageConfiguration configuration) throws IOException {
        ParametersChecker.checkParameter((String)"Workspace configuration cannot be null", (Object[])new Object[]{configuration});
        ParametersChecker.checkParameter((String)"Storage path configuration have to be define", (String[])new String[]{configuration.getStoragePath()});
        this.root = Paths.get(new File(configuration.getStoragePath()).getCanonicalPath(), new String[0]);
        if (!Files.exists(this.root, new LinkOption[0])) {
            Files.createDirectories(this.root, new FileAttribute[0]);
        }
    }

    public File checkWorkspaceContainerSanity(String container) throws IllegalPathException {
        return SafeFileChecker.checkSafeDirPath((String)this.root.toString(), (String[])new String[]{container});
    }

    public File checkWorkspaceDirSanity(String container, String directory) throws IllegalPathException {
        ArrayList<String> paths = new ArrayList<String>();
        paths.add(container);
        Collections.addAll(paths, directory.split(LINUX_PATH_SEPARATOR));
        return SafeFileChecker.checkSafeDirPath((String)this.root.toString(), (String[])paths.toArray(new String[0]));
    }

    public void checkWorkspaceFileSanity(String containerName, String relativeObjectName) throws IllegalPathException {
        String fullRelativePath = containerName + LINUX_PATH_SEPARATOR + relativeObjectName;
        SafeFileChecker.checkSafeFilePath((String)this.root.toString(), (String[])fullRelativePath.split(LINUX_PATH_SEPARATOR));
    }

    @Override
    public void createContainer(String containerName) throws ContentAddressableStorageAlreadyExistException, ContentAddressableStorageServerException {
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_NAME_IS_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName});
        if (!this.isExistingContainer(containerName)) {
            try {
                Path containerPath = this.getContainerPath(containerName);
                Files.createDirectories(containerPath, new FileAttribute[0]);
            }
            catch (IOException ex) {
                throw new ContentAddressableStorageServerException((Throwable)ex);
            }
        } else {
            throw new ContentAddressableStorageAlreadyExistException("Container " + containerName + "already exists");
        }
    }

    @Override
    public void purgeContainer(String containerName) throws ContentAddressableStorageNotFoundException, ContentAddressableStorageServerException {
        if (!this.isExistingContainer(containerName)) {
            LOGGER.info(ErrorMessage.CONTAINER_NOT_FOUND.getMessage() + containerName);
            throw new ContentAddressableStorageNotFoundException(ErrorMessage.CONTAINER_NOT_FOUND.getMessage() + containerName);
        }
        try {
            Path containerPath = this.getContainerPath(containerName);
            try (Stream<Path> streams = Files.walk(containerPath, FileVisitOption.FOLLOW_LINKS);){
                streams.filter(path -> !containerPath.equals(path)).sorted(Comparator.reverseOrder()).map(Path::toFile).forEach(File::delete);
            }
        }
        catch (IOException ex) {
            throw new ContentAddressableStorageServerException((Throwable)ex);
        }
    }

    @Override
    public void purgeOldFilesInContainer(String containerName, TimeToLive timeToLive) throws ContentAddressableStorageException {
        if (!this.isExistingContainer(containerName)) {
            LOGGER.info(String.valueOf(ErrorMessage.CONTAINER_NOT_FOUND) + containerName);
            return;
        }
        try {
            Path folderPath = this.getContainerPath(containerName);
            Instant expirationInstant = Instant.now().minus(timeToLive.getValue(), timeToLive.getUnit());
            try (Stream<Path> streams = Files.walk(folderPath, FileVisitOption.FOLLOW_LINKS);){
                streams.filter(path -> {
                    try {
                        if (Files.isDirectory(path, new LinkOption[0])) {
                            return false;
                        }
                        FileTime lastModifiedTime = Files.getLastModifiedTime(path, new LinkOption[0]);
                        return expirationInstant.isAfter(lastModifiedTime.toInstant());
                    }
                    catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }).forEach(path -> {
                    try {
                        Files.delete(path);
                    }
                    catch (IOException e) {
                        LOGGER.warn("Could not delete file " + String.valueOf(path), (Throwable)e);
                    }
                });
            }
        }
        catch (IOException ex) {
            throw new ContentAddressableStorageException((Throwable)ex);
        }
    }

    @Override
    public void deleteContainer(String containerName, boolean recursive) throws ContentAddressableStorageNotFoundException, ContentAddressableStorageServerException {
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_NAME_IS_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName});
        try {
            Path containerPath = this.getContainerPath(containerName);
            if (!containerPath.toFile().exists()) {
                LOGGER.error(ErrorMessage.CONTAINER_NOT_FOUND.getMessage() + containerName);
                throw new ContentAddressableStorageNotFoundException(ErrorMessage.CONTAINER_NOT_FOUND.getMessage() + containerName);
            }
            if (recursive) {
                FileUtils.deleteDirectory((File)containerPath.toFile());
            } else {
                Files.delete(containerPath);
            }
        }
        catch (IOException ex) {
            throw new ContentAddressableStorageServerException((Throwable)ex);
        }
    }

    @Override
    public boolean isExistingContainer(String containerName) {
        Path containerPath = this.getContainerPath(containerName);
        return containerPath.toFile().exists() && containerPath.toFile().isDirectory();
    }

    @Override
    public void createFolder(String containerName, String folderName) throws ContentAddressableStorageNotFoundException, ContentAddressableStorageAlreadyExistException, ContentAddressableStorageServerException {
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_FOLDER_NAMES_ARE_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName, folderName});
        if (!this.isExistingContainer(containerName)) {
            LOGGER.error(ErrorMessage.CONTAINER_NOT_FOUND.getMessage() + containerName);
            throw new ContentAddressableStorageNotFoundException(ErrorMessage.CONTAINER_NOT_FOUND.getMessage() + containerName);
        }
        if (this.isExistingFolder(containerName, folderName)) {
            LOGGER.info(String.valueOf(ErrorMessage.FOLDER_ALREADY_EXIST) + folderName);
            throw new ContentAddressableStorageAlreadyExistException(ErrorMessage.FOLDER_ALREADY_EXIST.getMessage() + folderName);
        }
        try {
            Path folderPath = this.getFolderPath(containerName, folderName);
            Files.createDirectories(folderPath, new FileAttribute[0]);
        }
        catch (IOException ex) {
            throw new ContentAddressableStorageServerException((Throwable)ex);
        }
    }

    @Override
    public void deleteFolder(String containerName, String folderName) throws ContentAddressableStorageNotFoundException, ContentAddressableStorageServerException {
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_FOLDER_NAMES_ARE_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName, folderName});
        if (!this.isExistingFolder(containerName, folderName)) {
            LOGGER.error(ErrorMessage.FOLDER_NOT_FOUND.getMessage() + folderName);
            throw new ContentAddressableStorageNotFoundException(ErrorMessage.FOLDER_NOT_FOUND.getMessage() + folderName);
        }
        Path folderPath = null;
        try {
            folderPath = this.getFolderPath(containerName, folderName);
            Files.delete(folderPath);
        }
        catch (DirectoryNotEmptyException ex) {
            LOGGER.warn("Directory {} not empty, then we delete files and folder", (Object)(folderPath != null ? folderPath.toString() : ""));
            try (Stream<Path> streams = Files.walk(folderPath, FileVisitOption.FOLLOW_LINKS);){
                streams.sorted(Comparator.reverseOrder()).map(Path::toFile).forEach(File::delete);
            }
            catch (IOException exc) {
                throw new ContentAddressableStorageServerException((Throwable)exc);
            }
        }
        catch (IOException ex) {
            throw new ContentAddressableStorageServerException((Throwable)ex);
        }
    }

    @Override
    public boolean isExistingFolder(String containerName, String folderName) {
        Path folderPath = this.getObjectPath(containerName, folderName);
        return folderPath.toFile().exists() && folderPath.toFile().isDirectory();
    }

    @Override
    public List<URI> getListUriDigitalObjectFromFolder(String containerName, String folderName, final int limit) throws ContentAddressableStorageException {
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_NAME_IS_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName});
        ParametersChecker.checkParameter((String)ErrorMessage.FOLDER_NOT_FOUND.getMessage(), (String[])new String[]{folderName});
        if (limit <= 0) {
            throw new IllegalArgumentException("Limit must be greater than 0");
        }
        if (!this.isExistingContainer(containerName)) {
            throw new ContentAddressableStorageNotFoundException("Container " + containerName + " not found");
        }
        if (!this.isExistingFolder(containerName, folderName)) {
            LOGGER.debug(ErrorMessage.FOLDER_NOT_FOUND.getMessage() + folderName);
            return Collections.emptyList();
        }
        try {
            final Path folderPath = this.getFolderPath(containerName, folderName);
            final ArrayList<URI> list = new ArrayList<URI>();
            Files.walkFileTree(folderPath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(this){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                    if (attrs.isDirectory()) {
                        return FileVisitResult.CONTINUE;
                    }
                    String pathFile = file.toString().replace(String.valueOf(folderPath) + WorkspaceFileSystem.LINUX_PATH_SEPARATOR, "");
                    list.add(URI.create(URLEncoder.encode(pathFile, StandardCharsets.UTF_8)));
                    if (list.size() >= limit) {
                        return FileVisitResult.TERMINATE;
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
            return list;
        }
        catch (IOException ex) {
            throw new ContentAddressableStorageException((Throwable)ex);
        }
    }

    @Override
    public void uncompressObject(String containerName, String folderName, String archiveMimeType, InputStream inputStreamObject) throws ContentAddressableStorageException {
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_OBJECT_NAMES_ARE_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName, folderName});
        if (!this.isExistingContainer(containerName)) {
            throw new ContentAddressableStorageNotFoundException(ErrorMessage.CONTAINER_NOT_FOUND.getMessage());
        }
        if (inputStreamObject == null) {
            throw new ContentAddressableStorageException(ErrorMessage.STREAM_IS_NULL.getMessage());
        }
        if (this.isExistingFolder(containerName, folderName)) {
            LOGGER.error(ErrorMessage.FOLDER_ALREADY_EXIST.getMessage() + " : folderName " + folderName);
            throw new ContentAddressableStorageAlreadyExistException(ErrorMessage.FOLDER_ALREADY_EXIST.getMessage());
        }
        this.createFolder(containerName, folderName);
        this.extractArchiveInputStreamOnContainer(containerName, folderName, CommonMediaType.valueOf((String)archiveMimeType), inputStreamObject);
    }

    @Override
    public void putObject(String containerName, String objectName, InputStream stream) throws ContentAddressableStorageException {
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_OBJECT_NAMES_ARE_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName, objectName});
        if (!this.isExistingContainer(containerName)) {
            LOGGER.error(ErrorMessage.CONTAINER_NOT_FOUND.getMessage() + containerName);
            throw new ContentAddressableStorageNotFoundException(ErrorMessage.CONTAINER_NOT_FOUND.getMessage() + containerName);
        }
        boolean existingObject = false;
        if (this.isExistingObject(containerName, objectName)) {
            LOGGER.info(ErrorMessage.OBJECT_ALREADY_EXIST.getMessage() + objectName);
            existingObject = true;
        }
        Path filePath = null;
        try {
            filePath = this.getObjectPath(containerName, objectName);
            if (!existingObject) {
                Path parentPath = filePath.getParent();
                if (!parentPath.toFile().exists()) {
                    Files.createDirectories(parentPath, new FileAttribute[0]);
                }
                filePath = Files.createFile(filePath, new FileAttribute[0]);
            }
            Files.copy(stream, filePath, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException ex) {
            LOGGER.error("Try to rollback because of ", (Throwable)ex);
            try {
                Files.deleteIfExists(filePath);
            }
            catch (IOException exc) {
                LOGGER.error("Cannot rollback because of ", (Throwable)exc);
            }
            throw new ContentAddressableStorageException((Throwable)ex);
        }
    }

    @Override
    public void putAtomicObject(String containerName, String objectName, InputStream stream, long size) throws ContentAddressableStorageException {
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_OBJECT_NAMES_ARE_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName, objectName});
        if (!this.isExistingContainer(containerName)) {
            LOGGER.error(ErrorMessage.CONTAINER_NOT_FOUND.getMessage() + containerName);
            throw new ContentAddressableStorageNotFoundException(ErrorMessage.CONTAINER_NOT_FOUND.getMessage() + containerName);
        }
        Path tmpFilePath = null;
        try {
            Path filePath = this.getObjectPath(containerName, objectName);
            Path parentPath = filePath.getParent();
            Files.createDirectories(parentPath, new FileAttribute[0]);
            String uniqueId = String.valueOf(filePath.getFileName()) + ".{" + String.valueOf(GUIDFactory.newGUID()) + "}";
            tmpFilePath = filePath.resolveSibling(uniqueId);
            try (OutputStream outputStream = Files.newOutputStream(tmpFilePath, new OpenOption[0]);
                 ExactSizeInputStream input = new ExactSizeInputStream(stream, size);){
                IOUtils.copy((InputStream)input, (OutputStream)outputStream);
            }
            FileUtil.fsyncFile((Path)tmpFilePath);
            Files.createLink(filePath, tmpFilePath);
            FileUtil.fsyncFile((Path)filePath);
            Files.delete(tmpFilePath);
        }
        catch (IOException ex) {
            if (tmpFilePath != null) {
                FileUtils.deleteQuietly((File)tmpFilePath.toFile());
            }
            throw new ContentAddressableStorageException((Throwable)ex);
        }
    }

    @Override
    public Response getObject(String containerName, String objectName, Long chunkOffset, Long maxChunkSize) throws ContentAddressableStorageException {
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_OBJECT_NAMES_ARE_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName, objectName});
        if (chunkOffset == null && maxChunkSize != null) {
            throw new IllegalArgumentException("Max chunk size without chunk offset");
        }
        if (!this.isExistingContainer(containerName)) {
            throw new ContentAddressableStorageNotFoundException(ErrorMessage.CONTAINER_NOT_FOUND.getMessage() + containerName);
        }
        if (!this.isExistingObject(containerName, objectName)) {
            throw new ContentAddressableStorageNotFoundException(ErrorMessage.OBJECT_NOT_FOUND.getMessage() + objectName);
        }
        try {
            Path objectPath = this.getObjectPath(containerName, objectName);
            InputStream inputStream = this.openFile(objectPath, chunkOffset, maxChunkSize);
            long totalSize = Files.size(objectPath);
            long chunkSize = chunkOffset == null ? totalSize : (maxChunkSize == null ? totalSize - chunkOffset : Math.min(maxChunkSize, totalSize - chunkOffset));
            return new AbstractMockClient.FakeInboundResponse(Response.Status.OK, (Object)new ExactSizeInputStream(inputStream, chunkSize), MediaType.APPLICATION_OCTET_STREAM_TYPE, this.getXContentLengthHeader(totalSize, chunkSize));
        }
        catch (IOException ex) {
            throw new ContentAddressableStorageException((Throwable)ex);
        }
    }

    private InputStream openFile(Path objectPath, Long chunkOffset, Long maxChunkSize) throws IOException {
        if (chunkOffset != null) {
            SeekableByteChannel seekableByteChannel = null;
            try {
                seekableByteChannel = Files.newByteChannel(objectPath, StandardOpenOption.READ);
                seekableByteChannel.position(chunkOffset);
                BufferedInputStream inputStream = new BufferedInputStream(Channels.newInputStream(seekableByteChannel));
                if (maxChunkSize == null) {
                    return inputStream;
                }
                return new BoundedInputStream((InputStream)inputStream, maxChunkSize.longValue());
            }
            catch (Exception ex) {
                StreamUtils.closeSilently(seekableByteChannel);
                throw ex;
            }
        }
        return new BufferedInputStream(Files.newInputStream(objectPath, StandardOpenOption.READ));
    }

    @Override
    public void deleteObject(String containerName, String objectName) throws ContentAddressableStorageException {
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_OBJECT_NAMES_ARE_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName, objectName});
        if (!this.isExistingContainer(containerName)) {
            LOGGER.error(ErrorMessage.CONTAINER_NOT_FOUND.getMessage() + containerName);
            throw new ContentAddressableStorageNotFoundException(ErrorMessage.CONTAINER_NOT_FOUND.getMessage() + containerName);
        }
        if (!this.isExistingObject(containerName, objectName)) {
            LOGGER.error(ErrorMessage.OBJECT_NOT_FOUND.getMessage() + objectName + " in container '" + containerName + "'");
            throw new ContentAddressableStorageNotFoundException(ErrorMessage.OBJECT_NOT_FOUND.getMessage() + objectName);
        }
        try {
            Files.delete(this.getObjectPath(containerName, objectName));
        }
        catch (IOException ex) {
            throw new ContentAddressableStorageException((Throwable)ex);
        }
    }

    @Override
    public boolean isExistingObject(String containerName, String objectName) {
        Path objectPath = this.getObjectPath(containerName, objectName);
        return objectPath.toFile().exists() && objectPath.toFile().isFile();
    }

    @Override
    public String computeObjectDigest(String containerName, String objectName, DigestType algo) throws ContentAddressableStorageException {
        String string;
        block8: {
            ParametersChecker.checkParameter((String)ErrorMessage.ALGO_IS_A_MANDATORY_PARAMETER.getMessage(), (Object[])new Object[]{algo});
            InputStream stream = (InputStream)this.getObject(containerName, objectName, null, null).getEntity();
            try {
                Digest digest = new Digest(algo);
                digest.update(stream);
                string = digest.toString();
                if (stream == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (stream != null) {
                        try {
                            stream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException ex) {
                    throw new ContentAddressableStorageException((Throwable)ex);
                }
            }
            stream.close();
        }
        return string;
    }

    @Override
    public ContainerInformation getContainerInformation(String containerName) throws ContentAddressableStorageNotFoundException, ContentAddressableStorageServerException {
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_NAME_IS_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName});
        if (!this.isExistingContainer(containerName)) {
            LOGGER.error(ErrorMessage.CONTAINER_NOT_FOUND.getMessage() + containerName);
            throw new ContentAddressableStorageNotFoundException(ErrorMessage.CONTAINER_NOT_FOUND.getMessage() + containerName);
        }
        try {
            Path containerPath = this.getContainerPath(containerName);
            long usableSpace = Files.getFileStore(containerPath).getUsableSpace();
            ContainerInformation containerInformation = new ContainerInformation();
            containerInformation.setUsableSpace(usableSpace);
            return containerInformation;
        }
        catch (IOException ex) {
            throw new ContentAddressableStorageServerException((Throwable)ex);
        }
    }

    @Override
    public JsonNode getObjectInformation(String containerName, String objectName) throws ContentAddressableStorageException {
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_OBJECT_NAMES_ARE_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName, objectName});
        if (!this.isExistingContainer(containerName)) {
            LOGGER.error(ErrorMessage.CONTAINER_NOT_FOUND.getMessage() + containerName);
            throw new ContentAddressableStorageNotFoundException(ErrorMessage.CONTAINER_NOT_FOUND.getMessage() + containerName);
        }
        if (!this.isExistingObject(containerName, objectName)) {
            LOGGER.error(ErrorMessage.OBJECT_NOT_FOUND.getMessage() + objectName);
            throw new ContentAddressableStorageNotFoundException(ErrorMessage.OBJECT_NOT_FOUND.getMessage() + objectName);
        }
        try {
            Path objectPath = this.getObjectPath(containerName, objectName);
            ObjectNode objectInformation = JsonHandler.createObjectNode();
            objectInformation.put("size", Files.size(objectPath));
            objectInformation.put("object_name", objectName);
            objectInformation.put("container_name", containerName);
            return objectInformation;
        }
        catch (IOException ex) {
            throw new ContentAddressableStorageException((Throwable)ex);
        }
    }

    private Path getContainerPath(String containerName) {
        return Paths.get(this.root.toString(), containerName);
    }

    private Path getFolderPath(String containerName, String folder) {
        return this.getObjectPath(containerName, folder);
    }

    private Path getObjectPath(String containerName, String objectName) {
        return Paths.get(this.root.toString(), containerName, objectName);
    }

    @Override
    public Map<String, FileParams> getFilesWithParamsFromFolder(String containerName, String folderName) throws ContentAddressableStorageException {
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_NAME_IS_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName});
        ParametersChecker.checkParameter((String)ErrorMessage.FOLDER_NOT_FOUND.getMessage(), (String[])new String[]{folderName});
        if (!this.isExistingContainer(containerName)) {
            throw new ContentAddressableStorageNotFoundException("Container " + containerName + " not found");
        }
        if (!this.isExistingFolder(containerName, folderName)) {
            LOGGER.debug(ErrorMessage.FOLDER_NOT_FOUND.getMessage() + folderName);
            return new HashMap<String, FileParams>();
        }
        try {
            final Path folderPath = this.getFolderPath(containerName, folderName);
            final HashMap<String, FileParams> filesWithParamsMap = new HashMap<String, FileParams>();
            Files.walkFileTree(folderPath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(this){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                    if (attrs.isDirectory()) {
                        return FileVisitResult.CONTINUE;
                    }
                    FileParams fileParams = new FileParams();
                    String pathFile = file.toString().replace(String.valueOf(folderPath) + WorkspaceFileSystem.LINUX_PATH_SEPARATOR, "");
                    fileParams.setSize(file.toFile().length());
                    filesWithParamsMap.put(pathFile, fileParams);
                    return FileVisitResult.CONTINUE;
                }
            });
            return filesWithParamsMap;
        }
        catch (IOException ex) {
            throw new ContentAddressableStorageException((Throwable)ex);
        }
    }

    private void extractArchiveInputStreamOnContainer(String containerName, String folderName, MediaType archiverType, InputStream inputStreamObject) throws ContentAddressableStorageException {
        try (InputStream inputStreamClosable = StreamUtils.getRemainingReadOnCloseInputStream((InputStream)inputStreamObject);
             ArchiveInputStream archiveInputStream = new VitamArchiveStreamFactory().createArchiveInputStream(archiverType, inputStreamClosable);){
            ArchiveEntry entry;
            boolean isEmpty = true;
            boolean manifestFileFound = false;
            ArchiveEntryInputStream entryInputStream = new ArchiveEntryInputStream((InputStream)archiveInputStream);
            File folder = this.checkWorkspaceDirSanity(containerName, folderName);
            while ((entry = archiveInputStream.getNextEntry()) != null) {
                if (archiveInputStream.canReadEntryData(entry)) {
                    File file;
                    if (entry.isDirectory()) continue;
                    isEmpty = false;
                    String entryName = entry.getName();
                    try {
                        String[] subPaths = entryName.split(LINUX_PATH_SEPARATOR);
                        file = SafeFileChecker.checkSafeFilePath((String)folder.getPath(), (String[])subPaths);
                    }
                    catch (IllegalPathException e) {
                        throw new ZipFilesNameNotAllowedException(String.format("%s file or folder not allowed name: '", entryName + "'"), (Throwable)e);
                    }
                    FileUtils.forceMkdirParent((File)file);
                    Path target = file.toPath();
                    Files.copy((InputStream)entryInputStream, target, StandardCopyOption.REPLACE_EXISTING);
                    if (!manifestFileFound && this.isManifestFileName(entry.getName())) {
                        Files.move(target, target.resolveSibling("manifest.xml"), new CopyOption[0]);
                        manifestFileFound = true;
                    }
                }
                entryInputStream.setClosed(false);
            }
            if (isEmpty) {
                throw new ContentAddressableStorageCompressedFileException("File is empty");
            }
        }
        catch (IllegalPathException | IOException | ArchiveException e) {
            throw new ContentAddressableStorageException(e);
        }
    }

    private MultivaluedHashMap<String, Object> getXContentLengthHeader(long totalSize, long chunkSize) {
        MultivaluedHashMap headers = new MultivaluedHashMap();
        headers.put((Object)VitamHttpHeader.X_CONTENT_LENGTH.getName(), Collections.singletonList(totalSize));
        headers.put((Object)VitamHttpHeader.X_CHUNK_LENGTH.getName(), Collections.singletonList(chunkSize));
        return headers;
    }

    public void compress(String containerName, List<String> folderNames, String zipName, String outputContainer) throws IOException, ArchiveException {
        Path zip = this.getObjectPath(outputContainer, zipName);
        try (final ArchiveOutputStream archive = new ArchiveStreamFactory().createArchiveOutputStream("zip", (OutputStream)new FileOutputStream(zip.toString()));){
            for (String folderName : folderNames) {
                final Path target = this.getFolderPath(containerName, folderName);
                Files.walkFileTree(target, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(this){

                    @Override
                    public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                        if (!attrs.isDirectory()) {
                            ZipArchiveOutputStream zipArchiveOutputStream = (ZipArchiveOutputStream)archive;
                            zipArchiveOutputStream.setUseZip64(Zip64Mode.Always);
                            Path relativize = Paths.get(target.getParent().toString(), new String[0]).relativize(file);
                            ZipArchiveEntry entry = new ZipArchiveEntry(relativize.toString());
                            zipArchiveOutputStream.putArchiveEntry(entry);
                            try (BufferedInputStream input = new BufferedInputStream(new FileInputStream(file.toFile()));){
                                IOUtils.copy((InputStream)input, (OutputStream)zipArchiveOutputStream);
                            }
                            zipArchiveOutputStream.closeArchiveEntry();
                        }
                        return FileVisitResult.CONTINUE;
                    }
                });
            }
        }
    }

    private boolean isManifestFileName(String fileName) {
        return fileName.matches("^([a-zA-Z0-9_\\-]{0,56}[_-]{1}){0,1}(manifest.xml)\\b");
    }

    public int getWorkspaceFreeSpace() {
        File workspace = this.root.toFile();
        if (workspace.getTotalSpace() == 0L) {
            return 100;
        }
        return (int)((float)workspace.getUsableSpace() / (float)workspace.getTotalSpace() * 100.0f);
    }

    @Override
    public void moveObjects(String containerName, List<BulkMoveEntry> entries) throws ContentAddressableStorageException {
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_NAME_IS_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName});
        if (!this.isExistingContainer(containerName)) {
            LOGGER.error(ErrorMessage.CONTAINER_NOT_FOUND.getMessage() + containerName);
            throw new ContentAddressableStorageNotFoundException(ErrorMessage.CONTAINER_NOT_FOUND.getMessage() + containerName);
        }
        try {
            for (BulkMoveEntry entry : entries) {
                String sourcePath = entry.source();
                String destinationPath = entry.destination();
                this.checkWorkspaceFileSanity(containerName, sourcePath);
                this.checkWorkspaceFileSanity(containerName, destinationPath);
                Path sourceFilePath = this.getObjectPath(containerName, sourcePath);
                Path destinationFilePath = this.getObjectPath(containerName, destinationPath);
                if (!sourceFilePath.toFile().exists()) {
                    if (destinationFilePath.toFile().exists()) {
                        LOGGER.warn("this file already moved" + sourcePath);
                        continue;
                    }
                    LOGGER.error(ErrorMessage.OBJECT_NOT_FOUND.getMessage() + sourcePath);
                    throw new ContentAddressableStorageNotFoundException(ErrorMessage.OBJECT_NOT_FOUND.getMessage() + sourcePath);
                }
                if (destinationFilePath.toFile().exists()) {
                    throw new ContentAddressableStorageAlreadyExistException("Cannot move '" + sourcePath + "' to '" + destinationPath + "' Destination path already exists");
                }
                Path parentPath = destinationFilePath.getParent();
                if (!parentPath.toFile().exists()) {
                    Files.createDirectories(parentPath, new FileAttribute[0]);
                }
                Files.move(sourceFilePath, destinationFilePath, StandardCopyOption.ATOMIC_MOVE);
            }
        }
        catch (IllegalPathException | IOException ex) {
            throw new ContentAddressableStorageException(ex);
        }
    }
}

