/*
 * Decompiled with CFR 0.152.
 */
package fr.gouv.vitam.common.security;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.google.json.JsonSanitizer;
import fr.gouv.vitam.common.StringUtils;
import fr.gouv.vitam.common.exception.InvalidParseOperationException;
import fr.gouv.vitam.common.json.JsonHandler;
import fr.gouv.vitam.common.logging.SysErrLogger;
import fr.gouv.vitam.common.logging.VitamLogger;
import fr.gouv.vitam.common.logging.VitamLoggerFactory;
import fr.gouv.vitam.common.xml.SecureXMLFactoryUtils;
import jakarta.ws.rs.core.HttpHeaders;
import jakarta.ws.rs.core.MultivaluedMap;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import org.owasp.esapi.ESAPI;
import org.owasp.esapi.Validator;
import org.owasp.esapi.errors.IntrusionException;
import org.owasp.esapi.errors.ValidationException;
import org.owasp.esapi.reference.DefaultValidator;
import org.owasp.esapi.reference.validation.HTMLValidationRule;

public class SanityChecker {
    private static final VitamLogger LOGGER = VitamLoggerFactory.getInstance(SanityChecker.class);
    private static final String JSON_IS_NOT_VALID_FROM_SANITIZE_CHECK = "Json is not valid from Sanitize check";
    private static final int DEFAULT_LIMIT_PARAMETER_SIZE = 5000;
    private static final int DEFAULT_LIMIT_FIELD_SIZE = 10000000;
    private static final int DEFAULT_LIMIT_JSON_SIZE = 16000000;
    private static final int DEFAULT_LIMIT_JSONL_SIZE = 100000000;
    private static final int DEFAULT_LIMIT_JSONL_LINE_COUNT = 100000;
    private static final long DEFAULT_LIMIT_FILE_SIZE = 8000000000L;
    public static final String HTTP_PARAMETER_VALUE = "HTTPParameterValue";
    public static final String HTTP_PERSISTENT_ID_PARAMETER_VALUE = "HTTPPersistentIdentifierParameterValue";
    private static final String HTTP_PARAMETER_NAME = "HTTPParameterName";
    private static final String HTTP_PERSISTENT_ID_PARAMETER_NAME = "persistentIdentifier";
    private static final String HTTP_HEADER_NAME = "HTTPHeaderName";
    private static final String HTTP_HEADER_VALUE = "HTTPHeaderValue";
    private static final String HTTP_HEADER_SSL_CLIENT_CERT_VALUE = "HTTPHeaderSslClientCertValue";
    private static final int REQUEST_LIMIT = 10000;
    private static long limitFileSize = 8000000000L;
    private static long limitJsonSize = 16000000L;
    private static long limitJsonLinesSize = 100000000L;
    private static long limitJsonLinesLineCount = 100000L;
    private static int limitFieldSize = 10000000;
    private static int limitParamSize = 5000;
    private static final Validator ESAPI = new SanityValidator();

    private SanityChecker() {
    }

    public static void checkXmlAll(File xmlFile) throws InvalidParseOperationException, IOException {
        SanityChecker.checkXmlSanityFileSize(xmlFile);
        SanityChecker.checkXmlSanityTags(xmlFile);
        SanityChecker.checkXmlSanityTagValueSize(xmlFile);
    }

    public static String sanitizeJson(JsonNode json) throws InvalidParseOperationException {
        if (json == null) {
            return "";
        }
        String jsonish = JsonHandler.writeAsString((Object)json);
        try {
            return JsonSanitizer.sanitize((String)jsonish);
        }
        catch (RuntimeException e) {
            throw new InvalidParseOperationException(JSON_IS_NOT_VALID_FROM_SANITIZE_CHECK, (Throwable)e);
        }
    }

    public static void checkJsonAll(JsonNode json) throws InvalidParseOperationException {
        if (json == null || json.isMissingNode()) {
            throw new InvalidParseOperationException(JSON_IS_NOT_VALID_FROM_SANITIZE_CHECK);
        }
        String jsonish = JsonHandler.writeAsString((Object)json);
        try {
            String wellFormedJson = JsonSanitizer.sanitize((String)jsonish);
            if (!wellFormedJson.equals(jsonish)) {
                throw new InvalidParseOperationException(JSON_IS_NOT_VALID_FROM_SANITIZE_CHECK);
            }
        }
        catch (RuntimeException e) {
            throw new InvalidParseOperationException(JSON_IS_NOT_VALID_FROM_SANITIZE_CHECK, (Throwable)e);
        }
        SanityChecker.checkJsonFileSize(jsonish);
        SanityChecker.checkJsonSanity(json);
    }

    public static void checkJsonLines(File jsonlFile) throws IOException, InvalidParseOperationException {
        long fileLength = jsonlFile.length();
        if (fileLength > limitJsonLinesSize) {
            LOGGER.error("Json-lines files too long " + fileLength + ". Max=" + limitJsonLinesSize);
            throw new InvalidParseOperationException("Max jsonl file size exceeded.");
        }
        try (BufferedReader bufferedReader = Files.newBufferedReader(jsonlFile.toPath(), StandardCharsets.UTF_8);){
            String line;
            int i = 0;
            while (null != (line = bufferedReader.readLine())) {
                if ((long)i > limitJsonLinesLineCount) {
                    throw new InvalidParseOperationException("Max jsonl line count exceeded.");
                }
                SanityChecker.checkJsonAll(line);
                ++i;
            }
        }
    }

    public static void checkJsonFile(File jsonlFile) throws IOException, InvalidParseOperationException {
        StringBuilder jsonFileContent = new StringBuilder();
        try (BufferedReader reader = new BufferedReader(new FileReader(jsonlFile, StandardCharsets.UTF_8));){
            String line;
            while ((line = reader.readLine()) != null) {
                jsonFileContent.append(line);
            }
        }
        SanityChecker.checkJsonAll(jsonFileContent.toString());
    }

    public static void checkJsonAll(String json) throws InvalidParseOperationException {
        try {
            String wellFormedJson = JsonSanitizer.sanitize((String)json);
            if (!wellFormedJson.equals(json)) {
                throw new InvalidParseOperationException(JSON_IS_NOT_VALID_FROM_SANITIZE_CHECK);
            }
        }
        catch (RuntimeException e) {
            throw new InvalidParseOperationException(JSON_IS_NOT_VALID_FROM_SANITIZE_CHECK, (Throwable)e);
        }
        SanityChecker.checkJsonFileSize(json);
        SanityChecker.checkJsonSanity(JsonHandler.getFromString((String)json));
    }

    public static void checkParameter(String ... params) throws InvalidParseOperationException {
        for (String param : params) {
            SanityChecker.checkParam(param);
        }
    }

    public static void checkHTMLFile(File file) throws InvalidParseOperationException, IOException {
        try (FileReader fileReader = new FileReader(file);
             BufferedReader bufReader = new BufferedReader(fileReader);){
            String line;
            while ((line = bufReader.readLine()) != null) {
                SanityChecker.checkParameter(line.split(","));
            }
        }
    }

    public static void checkHeaders(HttpHeaders headers) throws InvalidParseOperationException {
        if (headers == null) {
            return;
        }
        SanityChecker.checkHeadersMap((MultivaluedMap<String, String>)headers.getRequestHeaders());
    }

    public static void checkHeadersMap(MultivaluedMap<String, String> requestHeaders) throws InvalidParseOperationException {
        if (requestHeaders != null && !requestHeaders.isEmpty()) {
            for (String header : requestHeaders.keySet()) {
                if (SanityChecker.isStringInfected(header, HTTP_HEADER_NAME)) {
                    throw new InvalidParseOperationException(String.format("%s header has wrong name", header));
                }
                if (header.equals("X-SSL-CLIENT-CERT")) {
                    SanityChecker.validateSslClientCertHeader(requestHeaders, header);
                    continue;
                }
                List values = (List)requestHeaders.get((Object)header);
                if (values == null || !values.stream().anyMatch(value -> SanityChecker.isStringInfected(value, HTTP_HEADER_VALUE) | SanityChecker.isIssueOnParam(value))) continue;
                throw new InvalidParseOperationException(String.format("%s header has wrong value", header));
            }
        }
    }

    private static void validateSslClientCertHeader(MultivaluedMap<String, String> requestHeaders, String header) throws InvalidParseOperationException {
        List values = (List)requestHeaders.get((Object)header);
        if (values.size() > 1) {
            throw new InvalidParseOperationException(String.format("Multiple %s headers detected. SSL Certificate injection attack?", header));
        }
        if (SanityChecker.isStringInfected((String)requestHeaders.getFirst((Object)header), HTTP_HEADER_SSL_CLIENT_CERT_VALUE, false)) {
            throw new InvalidParseOperationException(String.format("%s header has wrong value", header));
        }
    }

    public static void checkUriParametersMap(MultivaluedMap<String, String> uriParameters) throws InvalidParseOperationException {
        if (uriParameters != null && !uriParameters.isEmpty()) {
            for (String parameter : uriParameters.keySet()) {
                if (SanityChecker.isStringInfected(parameter, HTTP_PARAMETER_NAME)) {
                    throw new InvalidParseOperationException(String.format("%s parameter has wrong name", parameter));
                }
                boolean isPersistentIdentifierParam = HTTP_PERSISTENT_ID_PARAMETER_NAME.equals(parameter);
                List values = (List)uriParameters.get((Object)parameter);
                if (values == null || !values.stream().anyMatch(value -> SanityChecker.isStringInfected(value, isPersistentIdentifierParam ? HTTP_PERSISTENT_ID_PARAMETER_VALUE : HTTP_PARAMETER_VALUE) | SanityChecker.isIssueOnParam(value))) continue;
                throw new InvalidParseOperationException(String.format("%s parameter has wrong value", parameter));
            }
        }
    }

    private static boolean isStringInfected(String value, String validator) {
        return SanityChecker.isStringInfected(value, validator, true);
    }

    private static boolean isStringInfected(String value, String validator, boolean canonicalize) {
        return !ESAPI.isValidInput(validator, value, validator, 10000, true, canonicalize);
    }

    private static boolean isIssueOnParam(String param) {
        try {
            SanityChecker.checkParam(param);
            return false;
        }
        catch (InvalidParseOperationException e) {
            SysErrLogger.FAKE_LOGGER.ignoreLog((Throwable)e);
            return true;
        }
    }

    private static void checkParam(String param) throws InvalidParseOperationException {
        SanityChecker.checkSanityTags(param, SanityChecker.getLimitParamSize());
        SanityChecker.checkHtmlPattern(param);
    }

    protected static void checkXmlSanityTagValueSize(File xmlFile) throws InvalidParseOperationException, IOException {
        try (FileInputStream xmlStream = new FileInputStream(xmlFile);){
            XMLStreamReader reader = null;
            try {
                reader = SecureXMLFactoryUtils.createSecureXMLStreamReader(xmlStream);
                while (reader.hasNext()) {
                    String val;
                    int event = reader.next();
                    if (event == 12 || event == 15 || event == 9) {
                        throw new InvalidParseOperationException("XML contains CDATA or ENTITY");
                    }
                    if (event != 4 || (val = reader.getText().trim()).isEmpty()) continue;
                    SanityChecker.checkSanityTags(val, SanityChecker.getLimitFieldSize());
                }
            }
            catch (XMLStreamException e) {
                throw new InvalidParseOperationException("Bad XML format", (Throwable)e);
            }
            finally {
                if (reader != null) {
                    try {
                        reader.close();
                    }
                    catch (XMLStreamException e) {
                        SysErrLogger.FAKE_LOGGER.ignoreLog((Throwable)e);
                    }
                }
            }
        }
    }

    protected static void checkXmlSanityFileSize(File xmlFile) throws InvalidParseOperationException {
        if (xmlFile.length() > SanityChecker.getLimitFileSize()) {
            throw new InvalidParseOperationException("File size exceeds sanity check");
        }
    }

    protected static void checkXmlSanityTags(File xmlFile) throws InvalidParseOperationException, IOException {
        try (FileReader fileReader = new FileReader(xmlFile);
             BufferedReader bufReader = new BufferedReader(fileReader);){
            String line;
            while ((line = bufReader.readLine()) != null) {
                SanityChecker.checkXmlSanityTags(line);
            }
        }
    }

    private static void checkXmlSanityTags(String line) throws InvalidParseOperationException {
        for (String rule : StringUtils.RULES) {
            SanityChecker.checkSanityTags(line, rule);
        }
    }

    private static void checkSanityTags(String line, int limit) throws InvalidParseOperationException {
        SanityChecker.checkSanityEsapi(line, limit);
        SanityChecker.checkXmlSanityTags(line);
    }

    private static void checkSanityEsapi(String line, int limit) throws InvalidParseOperationException {
        if (line.length() > limit) {
            throw new InvalidParseOperationException("Invalid input bytes length");
        }
        if (StringUtils.UNPRINTABLE_PATTERN.matcher(line).find()) {
            throw new InvalidParseOperationException("Invalid input bytes");
        }
        try {
            ESAPI.getValidSafeHTML("CheckSafeHtml", line, limit, true);
        }
        catch (NoClassDefFoundError | IntrusionException | ValidationException e) {
            throw new InvalidParseOperationException("Invalid ESAPI sanity check", e);
        }
    }

    private static void checkSanityTags(String dataLine, String invalidTag) throws InvalidParseOperationException {
        if (dataLine != null && invalidTag != null && dataLine.contains(invalidTag)) {
            throw new InvalidParseOperationException("Invalid tag sanity check");
        }
    }

    private static void checkHtmlPattern(String param) throws InvalidParseOperationException {
        if (StringUtils.HTML_PATTERN.matcher(param).find()) {
            throw new InvalidParseOperationException("HTML PATTERN found");
        }
    }

    protected static void checkJsonSanity(JsonNode json) throws InvalidParseOperationException {
        if (json.isArray()) {
            ArrayNode nodes = (ArrayNode)json;
            for (JsonNode element : nodes) {
                SanityChecker.checkJsonSanity(element);
            }
        } else {
            Iterator fields = json.fields();
            while (fields.hasNext()) {
                Map.Entry entry = (Map.Entry)fields.next();
                String key = (String)entry.getKey();
                SanityChecker.checkSanityTags(key, SanityChecker.getLimitFieldSize());
                JsonNode value = (JsonNode)entry.getValue();
                if (value.isArray()) {
                    ArrayNode nodes = (ArrayNode)value;
                    for (JsonNode jsonNode : nodes) {
                        if (!jsonNode.isValueNode()) {
                            SanityChecker.checkJsonSanity(jsonNode);
                            continue;
                        }
                        SanityChecker.validateJSONField(jsonNode);
                    }
                    continue;
                }
                if (!value.isValueNode()) {
                    SanityChecker.checkJsonSanity(value);
                    continue;
                }
                SanityChecker.validateJSONField(value);
            }
        }
    }

    private static void validateJSONField(JsonNode value) throws InvalidParseOperationException {
        String svalue = value.asText();
        SanityChecker.checkSanityTags(svalue, SanityChecker.getLimitFieldSize());
        SanityChecker.checkHtmlPattern(svalue);
    }

    private static void checkJsonFileSize(String json) throws InvalidParseOperationException {
        if ((long)json.length() > SanityChecker.getLimitJsonSize()) {
            throw new InvalidParseOperationException("Json size exceeds sanity check : " + SanityChecker.getLimitJsonSize());
        }
    }

    public static long getLimitFileSize() {
        return limitFileSize;
    }

    public static void setLimitFileSize(long limitFileSize) {
        SanityChecker.limitFileSize = limitFileSize;
    }

    public static long getLimitJsonSize() {
        return limitJsonSize;
    }

    public static void setLimitJsonSize(long limitJsonSize) {
        SanityChecker.limitJsonSize = limitJsonSize;
    }

    public static int getLimitFieldSize() {
        return limitFieldSize;
    }

    public static void setLimitFieldSize(int limitFieldSize) {
        SanityChecker.limitFieldSize = limitFieldSize;
    }

    public static int getLimitParamSize() {
        return limitParamSize;
    }

    public static void setLimitParamSize(int limitParamSize) {
        SanityChecker.limitParamSize = limitParamSize;
    }

    private static class SanityValidator
    extends DefaultValidator {
        private SanityValidator() {
        }

        public String getValidSafeHTML(String context, String input, int maxLength, boolean allowNull) throws ValidationException, IntrusionException {
            HTMLValidationRule hvr = new HTMLValidationRule("safehtml", org.owasp.esapi.ESAPI.encoder());
            hvr.setMaximumLength(maxLength);
            hvr.setAllowNull(allowNull);
            hvr.setCanonicalize(false);
            return hvr.getValid(context, input);
        }
    }
}

