/*
 *
 *  * Copyright French Prime minister Office/SGMAP/DINSIC/Vitam Program (2015-2022)
 *  *
 *  * contact.vitam@culture.gouv.fr
 *  *
 *  * This software is a computer program whose purpose is to implement a digital archiving back-office system managing
 *  * high volumetry securely and efficiently.
 *  *
 *  * This software is governed by the CeCILL 2.1 license under French law and abiding by the rules of distribution of free
 *  * software. You can use, modify and/ or redistribute the software under the terms of the CeCILL 2.1 license as
 *  * circulated by CEA, CNRS and INRIA at the following URL "https://cecill.info".
 *  *
 *  * As a counterpart to the access to the source code and rights to copy, modify and redistribute granted by the license,
 *  * users are provided only with a limited warranty and the software's author, the holder of the economic rights, and the
 *  * successive licensors have only limited liability.
 *  *
 *  * In this respect, the user's attention is drawn to the risks associated with loading, using, modifying and/or
 *  * developing or reproducing the software by the user in light of its specific status of free software, that may mean
 *  * that it is complicated to manipulate, and that also therefore means that it is reserved for developers and
 *  * experienced professionals having in-depth computer knowledge. Users are therefore encouraged to load and test the
 *  * software's suitability as regards their requirements in conditions enabling the security of their systems and/or data
 *  * to be ensured and, more generally, to use and operate it in the same conditions as regards security.
 *  *
 *  * The fact that you are presently reading this means that you have had knowledge of the CeCILL 2.1 license and that you
 *  * accept its terms.
 *
 */

package fr.gouv.vitamui.collect.server.service;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import fr.gouv.vitam.collect.common.dto.TransactionDto;
import fr.gouv.vitam.common.client.VitamContext;
import fr.gouv.vitam.common.exception.InvalidParseOperationException;
import fr.gouv.vitam.common.exception.VitamClientException;
import fr.gouv.vitam.common.json.JsonHandler;
import fr.gouv.vitam.common.model.RequestResponse;
import fr.gouv.vitam.common.model.RequestResponseOK;
import fr.gouv.vitamui.archives.search.common.dto.ReclassificationCriteriaDto;
import fr.gouv.vitamui.collect.common.dto.CollectTransactionDto;
import fr.gouv.vitamui.collect.server.service.converters.TransactionConverter;
import fr.gouv.vitamui.commons.api.exception.BadRequestException;
import fr.gouv.vitamui.commons.api.exception.InternalServerException;
import fr.gouv.vitamui.commons.api.exception.RequestTimeOutException;
import fr.gouv.vitamui.commons.vitam.api.collect.CollectService;
import lombok.RequiredArgsConstructor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.ws.rs.core.Response;
import java.io.InputStream;
import java.util.Arrays;

import static fr.gouv.vitamui.commons.api.utils.MetadataSearchCriteriaUtils.mapRequestToDslQuery;

@Service
@RequiredArgsConstructor
public class TransactionService {

    private static final Logger LOGGER = LoggerFactory.getLogger(TransactionService.class);

    public static final String UNABLE_TO_UPDATE_TRANSACTION = "Unable to update transaction";
    public static final String UNABLE_TO_PROCESS_RESPONSE = "Unable to process response";
    public static final String UNABLE_TO_PROCESS_UNIT_UPDATE = "Unable to process units update operation";
    public static final String ERROR_400 = "ERROR_400";
    public static final String REQUEST_TIMEOUT_EXCEPTION_MESSAGE =
        "the server has decided to close the connection rather than continue waiting";

    public static final String DSL_QUERY_PROJECTION = "$projection";
    public static final String DSL_QUERY_FILTER = "$filter";
    public static final String DSL_QUERY_THRESHOLD = "$threshold";
    public static final String DSL_QUERY_FACETS = "$facets";
    public static final String OPERATION_IDENTIFIER = "itemId";

    private static final String ACTION = "$action";

    private final CollectService collectService;

    public void validateTransaction(String idTransaction, VitamContext vitamContext) throws VitamClientException {
        try {
            RequestResponse requestResponse = collectService.validateTransaction(vitamContext, idTransaction);
            if (requestResponse.getStatus() != Response.Status.OK.getStatusCode()) {
                throw new VitamClientException("Error occurs when validating transaction!");
            }
        } catch (VitamClientException e) {
            throw new VitamClientException("Unable to validate transaction : ", e);
        }
    }

    public void sendTransaction(String idTransaction, VitamContext vitamContext) throws VitamClientException {
        try {
            RequestResponse requestResponse = collectService.sendTransaction(vitamContext, idTransaction);
            if (requestResponse.getStatus() != Response.Status.OK.getStatusCode()) {
                throw new VitamClientException("Error occurs when sending transaction!");
            }
        } catch (VitamClientException e) {
            throw new VitamClientException("Unable to send transaction : ", e);
        }
    }

    public void abortTransaction(String idTransaction, VitamContext vitamContext) throws VitamClientException {
        try {
            RequestResponse requestResponse = collectService.abortTransaction(vitamContext, idTransaction);
            if (requestResponse.getStatus() != Response.Status.OK.getStatusCode()) {
                throw new VitamClientException("Error occurs when aborting transaction!");
            }
        } catch (VitamClientException e) {
            throw new VitamClientException("Unable to abort transaction : ", e);
        }
    }

    public void reopenTransaction(String idTransaction, VitamContext vitamContext) throws VitamClientException {
        try {
            RequestResponse requestResponse = collectService.reopenTransaction(vitamContext, idTransaction);
            if (requestResponse.getStatus() != Response.Status.OK.getStatusCode()) {
                throw new VitamClientException("Error occurs when reopening transaction!");
            }
        } catch (VitamClientException e) {
            throw new VitamClientException("Unable to reopen transaction : ", e);
        }
    }

    public CollectTransactionDto getTransactionById(String transactionId, VitamContext vitamContext)
        throws VitamClientException {
        try {
            RequestResponse<JsonNode> requestResponse = collectService.getTransactionById(vitamContext, transactionId);
            if (!requestResponse.isOk()) {
                throw new VitamClientException("Error occurs when getting transaction!");
            }

            return TransactionConverter.toVitamUiDto(
                JsonHandler.getFromString(
                    ((RequestResponseOK) requestResponse).getFirstResult().toString(),
                    TransactionDto.class
                )
            );
        } catch (VitamClientException | InvalidParseOperationException e) {
            throw new VitamClientException("Unable to find transaction : ", e);
        }
    }

    public CollectTransactionDto updateTransaction(
        CollectTransactionDto collectTransactionDto,
        VitamContext vitamContext
    ) {
        LOGGER.debug("CollectTransactionDto: ", collectTransactionDto);
        try {
            TransactionDto transactionDto = TransactionConverter.toVitamDto(collectTransactionDto);
            RequestResponse<JsonNode> requestResponse = collectService.updateTransaction(vitamContext, transactionDto);
            if (!requestResponse.isOk()) {
                throw new VitamClientException("Error occurs when updating transaction!");
            }
            TransactionDto responseTransactionDto = JsonHandler.getFromString(
                ((RequestResponseOK) requestResponse).getFirstResult().toString(),
                TransactionDto.class
            );
            return TransactionConverter.toVitamUiDto(responseTransactionDto);
        } catch (VitamClientException e) {
            LOGGER.debug(UNABLE_TO_UPDATE_TRANSACTION + ": {}", e);
            throw new InternalServerException(UNABLE_TO_UPDATE_TRANSACTION, e);
        } catch (InvalidParseOperationException e) {
            LOGGER.debug(UNABLE_TO_PROCESS_RESPONSE + ": {}", e);
            throw new InternalServerException(UNABLE_TO_PROCESS_RESPONSE, e);
        }
    }

    public String updateArchiveUnitsFromFile(InputStream inputStream, String transactionId, VitamContext vitamContext)
        throws RequestTimeOutException {
        LOGGER.debug("[Internal] call update Archive Units From File for transaction Id {}  ", transactionId);
        final String result = collectService.updateCollectArchiveUnits(vitamContext, transactionId, inputStream);
        if (result.equals(ERROR_400)) {
            LOGGER.debug(UNABLE_TO_PROCESS_UNIT_UPDATE);
            throw new RequestTimeOutException(REQUEST_TIMEOUT_EXCEPTION_MESSAGE, REQUEST_TIMEOUT_EXCEPTION_MESSAGE);
        }
        return result;
    }

    public String reclassification(
        final String transactionId,
        final ReclassificationCriteriaDto reclassificationCriteriaDto,
        final VitamContext vitamContext
    ) throws VitamClientException {
        if (reclassificationCriteriaDto == null) {
            throw new BadRequestException("Error reclassification criteria");
        }
        LOGGER.debug("Reclassification Object : {}", reclassificationCriteriaDto.toString());
        JsonNode dslQuery = mapRequestToDslQuery(reclassificationCriteriaDto.getSearchCriteriaDto());
        ArrayNode array = JsonHandler.createArrayNode();
        ((ObjectNode) dslQuery).putPOJO(ACTION, reclassificationCriteriaDto.getAction());
        Arrays.stream(
            new String[] { DSL_QUERY_PROJECTION, DSL_QUERY_FILTER, DSL_QUERY_FACETS, DSL_QUERY_THRESHOLD }
        ).forEach(((ObjectNode) dslQuery)::remove);
        array.add(dslQuery);
        LOGGER.debug("Reclassification query : {}", array);
        RequestResponse<JsonNode> jsonNodeRequestResponse = collectService.reclassification(
            vitamContext,
            transactionId,
            array
        );
        return jsonNodeRequestResponse.toJsonNode().findValue(OPERATION_IDENTIFIER).textValue();
    }
}
