/*
 * Copyright French Prime minister Office/SGMAP/DINSIC/Vitam Program (2015-2022)
 *
 * contact.vitam@culture.gouv.fr
 *
 * This software is a computer program whose purpose is to implement a digital archiving back-office system managing
 * high volumetry securely and efficiently.
 *
 * This software is governed by the CeCILL 2.1 license under French law and abiding by the rules of distribution of free
 * software. You can use, modify and/ or redistribute the software under the terms of the CeCILL 2.1 license as
 * circulated by CEA, CNRS and INRIA at the following URL "https://cecill.info".
 *
 * As a counterpart to the access to the source code and rights to copy, modify and redistribute granted by the license,
 * users are provided only with a limited warranty and the software's author, the holder of the economic rights, and the
 * successive licensors have only limited liability.
 *
 * In this respect, the user's attention is drawn to the risks associated with loading, using, modifying and/or
 * developing or reproducing the software by the user in light of its specific status of free software, that may mean
 * that it is complicated to manipulate, and that also therefore means that it is reserved for developers and
 * experienced professionals having in-depth computer knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions enabling the security of their systems and/or data
 * to be ensured and, more generally, to use and operate it in the same conditions as regards security.
 *
 * The fact that you are presently reading this means that you have had knowledge of the CeCILL 2.1 license and that you
 * accept its terms.
 */
package fr.gouv.vitamui.collect.server.rest;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import fr.gouv.vitam.common.database.builder.request.exception.InvalidCreateOperationException;
import fr.gouv.vitam.common.exception.InvalidParseOperationException;
import fr.gouv.vitam.common.exception.VitamClientException;
import fr.gouv.vitamui.archives.search.common.dto.VitamUIArchiveUnitResponseDto;
import fr.gouv.vitamui.archives.search.common.rest.RestApi;
import fr.gouv.vitamui.collect.server.service.ExternalParametersService;
import fr.gouv.vitamui.collect.server.service.TransactionArchiveUnitService;
import fr.gouv.vitamui.common.security.SanityChecker;
import fr.gouv.vitamui.commons.api.CommonConstants;
import fr.gouv.vitamui.commons.api.ParameterChecker;
import fr.gouv.vitamui.commons.api.domain.ServicesData;
import fr.gouv.vitamui.commons.api.dtos.SearchCriteriaDto;
import fr.gouv.vitamui.commons.api.dtos.VitamUiOntologyDto;
import fr.gouv.vitamui.commons.api.exception.PreconditionFailedException;
import fr.gouv.vitamui.commons.vitam.api.dto.ResultsDto;
import fr.gouv.vitamui.iam.security.service.SecurityService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.security.access.annotation.Secured;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import javax.ws.rs.Consumes;
import java.io.IOException;
import java.util.List;

import static fr.gouv.vitamui.archives.search.common.rest.RestApi.DELETION_ACTION;
import static fr.gouv.vitamui.archives.search.common.rest.RestApi.EXPORT_CSV_SEARCH_PATH;
import static fr.gouv.vitamui.collect.common.rest.RestApi.ARCHIVE_UNITS;
import static fr.gouv.vitamui.collect.common.rest.RestApi.COLLECT_TRANSACTION_ARCHIVE_UNITS_PATH;

/**
 * Project Archive units External controller
 */
@Api(tags = "Collect")
@RequestMapping(COLLECT_TRANSACTION_ARCHIVE_UNITS_PATH)
@RestController
@ResponseBody
@RequiredArgsConstructor
public class TransactionArchiveUnitController {

    private static final Logger LOGGER = LoggerFactory.getLogger(TransactionArchiveUnitController.class);

    private static final String MANDATORY_QUERY = "The query is a mandatory parameter: ";
    private static final String MANDATORY_IDENTIFIER = "The identifier is a mandatory parameter: ";

    private final TransactionArchiveUnitService transactionArchiveUnitService;
    private final ExternalParametersService externalParametersService;
    private final SecurityService securityService;

    @ApiOperation(value = "find archive units by criteria")
    @Secured(ServicesData.ROLE_GET_PROJECTS)
    @PostMapping("/{transactionId}" + ARCHIVE_UNITS)
    @Consumes(MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    public VitamUIArchiveUnitResponseDto searchArchiveUnits(
        final @PathVariable("transactionId") String transactionId,
        @RequestBody final SearchCriteriaDto searchQuery
    )
        throws PreconditionFailedException, InvalidCreateOperationException, VitamClientException, InvalidParseOperationException, JsonProcessingException {
        ParameterChecker.checkParameter(
            "The Query and the transactionId are mandatories parameters: ",
            transactionId,
            searchQuery
        );
        SanityChecker.sanitizeCriteria(searchQuery);
        SanityChecker.checkSecureParameter(transactionId);
        LOGGER.debug("search archives Units by criteria = {}", searchQuery);

        return transactionArchiveUnitService.searchArchiveUnitsByCriteria(
            transactionId,
            searchQuery,
            externalParametersService.buildVitamContextFromExternalParam()
        );
    }

    @PostMapping("/{transactionId}" + ARCHIVE_UNITS + EXPORT_CSV_SEARCH_PATH)
    @Secured(ServicesData.COLLECT_GET_ARCHIVE_SEARCH_ROLE)
    public Resource exportCsvArchiveUnitsByCriteria(
        final @PathVariable("transactionId") String transactionId,
        final @RequestBody SearchCriteriaDto query
    ) throws PreconditionFailedException, VitamClientException {
        ParameterChecker.checkParameter(MANDATORY_QUERY, query);
        SanityChecker.checkSecureParameter(transactionId);
        SanityChecker.sanitizeCriteria(query);
        LOGGER.debug("Calling export to csv search archive Units By Criteria {} ", query);
        return transactionArchiveUnitService.exportToCsvSearchArchiveUnitsByCriteria(
            transactionId,
            query,
            externalParametersService.buildVitamContextFromExternalParam()
        );
    }

    @GetMapping(RestApi.ARCHIVE_UNIT_INFO + CommonConstants.PATH_ID)
    @Secured(ServicesData.COLLECT_GET_ARCHIVE_SEARCH_ROLE)
    public ResultsDto findUnitById(final @PathVariable("id") String id)
        throws PreconditionFailedException, VitamClientException {
        ParameterChecker.checkParameter(MANDATORY_IDENTIFIER, id);
        SanityChecker.checkSecureParameter(id);
        LOGGER.debug("the UA by id {} ", id);
        return transactionArchiveUnitService.findArchiveUnitById(
            id,
            externalParametersService.buildVitamContextFromExternalParam()
        );
    }

    @GetMapping(CommonConstants.OBJECTS_PATH + CommonConstants.PATH_ID)
    @Secured(ServicesData.COLLECT_GET_ARCHIVE_SEARCH_ROLE)
    public ResultsDto getObjectGroupById(final @PathVariable("id") String objectId)
        throws PreconditionFailedException, VitamClientException {
        ParameterChecker.checkParameter(MANDATORY_IDENTIFIER, objectId);
        SanityChecker.checkSecureParameter(objectId);
        LOGGER.debug("[EXTERNAL] : Get ObjectGroup By id : {}", objectId);
        return transactionArchiveUnitService.findObjectGroupById(
            objectId,
            externalParametersService.buildVitamContextFromExternalParam()
        );
    }

    @GetMapping(CommonConstants.EXTERNAL_ONTOLOGIES_LIST)
    @Secured(ServicesData.COLLECT_GET_ARCHIVE_SEARCH_ROLE)
    public List<VitamUiOntologyDto> getExternalOntologiesList() throws IOException {
        LOGGER.debug("[EXTERNAL] : Get external ontologies list");
        return transactionArchiveUnitService.readExternalOntologiesFromFile(securityService.getTenantIdentifier());
    }

    @PostMapping("/{transactionId}" + RestApi.UNIT_WITH_INHERITED_RULES)
    @Secured(ServicesData.COLLECT_GET_ARCHIVE_SEARCH_ROLE)
    public ResultsDto selectUnitWithInheritedRules(
        final @PathVariable("transactionId") String transactionId,
        final @RequestBody SearchCriteriaDto query
    ) throws PreconditionFailedException, VitamClientException, IOException {
        ParameterChecker.checkParameter(MANDATORY_QUERY, query);
        SanityChecker.checkSecureParameter(transactionId);
        SanityChecker.sanitizeCriteria(query);
        LOGGER.debug("Calling select Unit With Inherited Rules By Criteria {} ", query);
        return transactionArchiveUnitService.selectUnitWithInheritedRules(
            query,
            transactionId,
            externalParametersService.buildVitamContextFromExternalParam()
        );
    }

    @Secured(ServicesData.COLLECT_DELETE_ARCHIVE_UNIT_ROLE)
    @ApiOperation(
        value = "Upload on streaming metadata file and update archive units",
        consumes = MediaType.APPLICATION_JSON_VALUE
    )
    @PostMapping(
        value = CommonConstants.TRANSACTION_PATH_ID + DELETION_ACTION,
        consumes = MediaType.APPLICATION_JSON_VALUE
    )
    public JsonNode startDeletionAction(
        final @PathVariable("transactionId") String transactionId,
        final @RequestBody SearchCriteriaDto query
    ) throws PreconditionFailedException, VitamClientException {
        ParameterChecker.checkParameter(MANDATORY_QUERY, query);
        SanityChecker.sanitizeCriteria(query);
        LOGGER.debug("Calling deletion action by criteria {} ", query);
        return transactionArchiveUnitService.startDeletionAction(
            externalParametersService.buildVitamContextFromExternalParam(),
            transactionId,
            query
        );
    }
}
