/**
 * Copyright French Prime minister Office/SGMAP/DINSIC/Vitam Program (2019-2020)
 * and the signatories of the "VITAM - Accord du Contributeur" agreement.
 *
 * contact@programmevitam.fr
 *
 * This software is a computer program whose purpose is to implement
 * implement a digital archiving front-office system for the secure and
 * efficient high volumetry VITAM solution.
 *
 * This software is governed by the CeCILL-C license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL-C
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided only
 * with a limited warranty  and the software's author,  the holder of the
 * economic rights,  and the successive licensors  have only  limited
 * liability.
 *
 * In this respect, the user's attention is drawn to the risks associated
 * with loading,  using,  modifying and/or developing or reproducing the
 * software by the user in light of its specific status of free software,
 * that may mean  that it is complicated to manipulate,  and  that  also
 * therefore means  that it is reserved for developers  and  experienced
 * professionals having in-depth computer knowledge. Users are therefore
 * encouraged to load and test the software's suitability as regards their
 * requirements in conditions enabling the security of their systems and/or
 * data to be ensured and,  more generally, to use and operate it in the
 * same conditions as regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL-C license and that you accept its terms.
 */
package fr.gouv.vitamui.iam.server.rest;

import fr.gouv.vitam.common.exception.InvalidParseOperationException;
import fr.gouv.vitam.common.exception.VitamClientException;
import fr.gouv.vitamui.common.security.SanityChecker;
import fr.gouv.vitamui.commons.api.CommonConstants;
import fr.gouv.vitamui.commons.api.ParameterChecker;
import fr.gouv.vitamui.commons.api.domain.DirectionDto;
import fr.gouv.vitamui.commons.api.domain.PaginatedValuesDto;
import fr.gouv.vitamui.commons.api.domain.ProfileDto;
import fr.gouv.vitamui.commons.api.domain.ServicesData;
import fr.gouv.vitamui.commons.api.exception.PreconditionFailedException;
import fr.gouv.vitamui.commons.api.utils.CastUtils;
import fr.gouv.vitamui.commons.api.utils.EnumUtils;
import fr.gouv.vitamui.commons.rest.CrudController;
import fr.gouv.vitamui.commons.rest.util.RestUtils;
import fr.gouv.vitamui.commons.vitam.api.dto.LogbookOperationsCommonResponseDto;
import fr.gouv.vitamui.iam.common.dto.common.EmbeddedOptions;
import fr.gouv.vitamui.iam.common.rest.RestApi;
import fr.gouv.vitamui.iam.security.service.SecurityService;
import fr.gouv.vitamui.iam.server.profile.service.ProfileService;
import io.swagger.annotations.Api;
import lombok.Getter;
import lombok.Setter;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.annotation.Secured;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.validation.Valid;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * The controller to check existence, create, read, update and delete the profiles.
 *
 *
 */
@RestController
@RequestMapping(RestApi.V1_PROFILES_URL)
@Getter
@Setter
@Api(tags = "profiles", value = "Profiles Management", description = "Profiles Management")
public class ProfileController implements CrudController<ProfileDto> {

    private static final Logger LOGGER = LoggerFactory.getLogger(ProfileController.class);

    protected static final String CUSTOMER_ID_KEY = "customerId";
    protected static final String TENANT_IDENTIFIER_KEY = "tenantIdentifier";
    protected static final String LEVEL_KEY = "level";

    private final ProfileService profileService;
    private final SecurityService securityService;

    @Autowired
    public ProfileController(final ProfileService profileService, final SecurityService securityService) {
        this.profileService = profileService;
        this.securityService = securityService;
    }

    @GetMapping
    @Secured(ServicesData.ROLE_GET_PROFILES)
    public Collection<ProfileDto> getAll(
        final Optional<String> criteria,
        @RequestParam final Optional<String> embedded
    ) {
        SanityChecker.sanitizeCriteria(criteria);
        EnumUtils.checkValidEnum(EmbeddedOptions.class, embedded);
        LOGGER.debug("Get all with criteria={}, embedded={}", criteria, embedded);
        return profileService.getAll(criteria, embedded);
    }

    @Override
    @Secured(ServicesData.ROLE_GET_PROFILES)
    @RequestMapping(path = CommonConstants.PATH_CHECK, method = RequestMethod.HEAD)
    public ResponseEntity<Void> checkExist(@RequestParam final String criteria) {
        SanityChecker.sanitizeCriteria(criteria);
        LOGGER.debug("checkExist criteria={}", criteria);
        final boolean exist = profileService.checkExist(criteria);
        return RestUtils.buildBooleanResponse(exist);
    }

    @GetMapping(CommonConstants.PATH_ID)
    @Secured(ServicesData.ROLE_GET_PROFILES)
    public ProfileDto getOne(final @PathVariable("id") String id, final @RequestParam Optional<String> embedded)
        throws InvalidParseOperationException, PreconditionFailedException {
        ParameterChecker.checkParameter("The Identifier is a mandatory parameter: ", id);
        SanityChecker.checkSecureParameter(id);
        EnumUtils.checkValidEnum(EmbeddedOptions.class, embedded);
        LOGGER.debug("Get {}, embedded={}", id, embedded);
        return profileService.getOne(id, Optional.empty(), embedded);
    }

    @PostMapping
    @Secured(ServicesData.ROLE_CREATE_PROFILES)
    @Override
    public ProfileDto create(final @Valid @RequestBody ProfileDto dto)
        throws InvalidParseOperationException, PreconditionFailedException {
        SanityChecker.sanitizeCriteria(dto);
        LOGGER.debug("Create {}", dto);
        profileService.checkCustomerId(dto.getCustomerId(), "Unable to create profile " + dto.getName());
        profileService.checkTenantIdentifier(dto.getTenantIdentifier(), "Unable to create profile");
        return profileService.create(dto);
    }

    @Override
    public ProfileDto update(final @PathVariable("id") String id, final @Valid @RequestBody ProfileDto dto) {
        throw new UnsupportedOperationException("update not implemented");
    }

    @Secured(ServicesData.ROLE_GET_PROFILES)
    @GetMapping(params = { "page", "size" })
    public PaginatedValuesDto<ProfileDto> getAllPaginated(
        @RequestParam final Integer page,
        @RequestParam final Integer size,
        @RequestParam(required = false) final Optional<String> criteria,
        @RequestParam(required = false) final Optional<String> orderBy,
        @RequestParam(required = false) final Optional<DirectionDto> direction,
        @RequestParam(required = false) final Optional<String> embedded
    ) throws InvalidParseOperationException, PreconditionFailedException {
        SanityChecker.checkSecureParameter(String.valueOf(size), String.valueOf(page));
        SanityChecker.sanitizeCriteria(criteria);
        direction.ifPresent(SanityChecker::sanitizeCriteria);
        orderBy.ifPresent(SanityChecker::checkSecureParameter);
        EnumUtils.checkValidEnum(EmbeddedOptions.class, embedded);
        LOGGER.debug(
            "getAllPaginated page={}, size={}, criteria={}, orderBy={}, direction={}, embedded = {}",
            page,
            size,
            criteria,
            orderBy,
            direction,
            embedded
        );
        return profileService.getAllPaginated(page, size, criteria, orderBy, direction, embedded);
    }

    @Override
    @PatchMapping(CommonConstants.PATH_ID)
    @Secured(ServicesData.ROLE_UPDATE_PROFILES)
    public ProfileDto patch(final @PathVariable("id") String id, @RequestBody final Map<String, Object> partialDto)
        throws InvalidParseOperationException, PreconditionFailedException {
        ParameterChecker.checkParameter("Identifier is mandatory : ", id);
        SanityChecker.checkSecureParameter(id);
        SanityChecker.sanitizeCriteria(partialDto);
        LOGGER.debug("Patch {} with {}", id, partialDto);
        Assert.isTrue(
            StringUtils.equals(id, (String) partialDto.get("id")),
            "The DTO identifier must match the path identifier for update."
        );
        final String customerId = CastUtils.toString(partialDto.get(CUSTOMER_ID_KEY));
        final Integer tenantIdentifier = CastUtils.toInteger(partialDto.get(TENANT_IDENTIFIER_KEY));
        final String level = CastUtils.toString(partialDto.get(LEVEL_KEY));
        if (StringUtils.isNotEmpty(customerId)) {
            profileService.checkCustomerId(customerId, "Unable to patch profile");
        }
        if (tenantIdentifier != null) {
            profileService.checkTenantIdentifier(tenantIdentifier, "Unable to patch profile");
        }
        if (StringUtils.isNotEmpty(level)) {
            profileService.checkLevel(level, "Unable to patch profile");
        }
        partialDto.put(CUSTOMER_ID_KEY, securityService.getCustomerId());
        partialDto.put(TENANT_IDENTIFIER_KEY, securityService.getTenantIdentifier());
        return profileService.patch(partialDto);
    }

    @GetMapping(CommonConstants.PATH_LOGBOOK)
    public LogbookOperationsCommonResponseDto findHistoryById(final @PathVariable("id") String id)
        throws VitamClientException, InvalidParseOperationException {
        ParameterChecker.checkParameter("The Identifier is a mandatory parameter: ", id);
        SanityChecker.checkSecureParameter(id);
        LOGGER.debug("get logbook for profile with id :{}", id);
        return profileService.findHistoryById(id);
    }

    /**
     * Get levels by criteria.
     * @param criteria Criteria as json string
     * @return List of matching levels
     */
    @GetMapping(CommonConstants.PATH_LEVELS)
    @Secured(ServicesData.ROLE_GET_PROFILES)
    public List<String> getLevels(final Optional<String> criteria) {
        SanityChecker.sanitizeCriteria(criteria);
        LOGGER.debug("Get levels with criteria={}", criteria);
        return profileService.getLevels(criteria);
    }
}
