/**
 * Copyright French Prime minister Office/SGMAP/DINSIC/Vitam Program (2019-2020)
 * and the signatories of the "VITAM - Accord du Contributeur" agreement.
 *
 * contact@programmevitam.fr
 *
 * This software is a computer program whose purpose is to implement
 * implement a digital archiving front-office system for the secure and
 * efficient high volumetry VITAM solution.
 *
 * This software is governed by the CeCILL-C license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL-C
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided only
 * with a limited warranty  and the software's author,  the holder of the
 * economic rights,  and the successive licensors  have only  limited
 * liability.
 *
 * In this respect, the user's attention is drawn to the risks associated
 * with loading,  using,  modifying and/or developing or reproducing the
 * software by the user in light of its specific status of free software,
 * that may mean  that it is complicated to manipulate,  and  that  also
 * therefore means  that it is reserved for developers  and  experienced
 * professionals having in-depth computer knowledge. Users are therefore
 * encouraged to load and test the software's suitability as regards their
 * requirements in conditions enabling the security of their systems and/or
 * data to be ensured and,  more generally, to use and operate it in the
 * same conditions as regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL-C license and that you accept its terms.
 */

package fr.gouv.vitamui.iam.server.rest;

import fr.gouv.vitamui.common.security.SanityChecker;
import fr.gouv.vitamui.commons.api.CommonConstants;
import fr.gouv.vitamui.commons.api.domain.ApplicationDto;
import fr.gouv.vitamui.commons.api.utils.EnumUtils;
import fr.gouv.vitamui.iam.common.dto.common.EmbeddedOptions;
import fr.gouv.vitamui.iam.common.rest.RestApi;
import fr.gouv.vitamui.iam.server.application.service.ApplicationService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.Getter;
import lombok.Setter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.validation.Valid;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * The controller to check list available applications.
 */
@RestController
@RequestMapping(RestApi.V1_APPLICATIONS_URL)
@Getter
@Setter
@Api(tags = "applications", value = "Applications Management", description = "Applications Management")
public class ApplicationController {

    private static final Logger LOGGER = LoggerFactory.getLogger(ApplicationController.class);

    private final ApplicationService applicationService;

    @Autowired
    public ApplicationController(final ApplicationService applicationService) {
        this.applicationService = applicationService;
    }

    /**
     * Get All applications filtered by user privileges
     *
     * @param criteria
     * @param embedded
     * @return all Applications matching user privileges
     */
    @GetMapping
    public List<ApplicationDto> getAll(final Optional<String> criteria, @RequestParam final Optional<String> embedded) {
        SanityChecker.sanitizeCriteria(criteria);
        EnumUtils.checkValidEnum(EmbeddedOptions.class, embedded);
        LOGGER.debug("Get all with criteria={}, embedded={}", criteria, embedded);
        return applicationService.getAll(criteria, embedded);
    }

    @GetMapping("/filtered")
    @ApiOperation(value = "Return config about applications and categories")
    public Map<String, Object> getApplicationsFromUi(@RequestParam(defaultValue = "true") final boolean filterApp) {
        LOGGER.debug("getApplications");
        return applicationService.getApplications(filterApp);
    }

    /**
     * /!\ Important - Invoked from both internal and external clients. If a role is added for external use, ROLE_INTERNAL must also be added.
     */
    @GetMapping(path = "/{identifier:.+}/externalid")
    @ApiOperation(value = "Check if an application can have an external identifier")
    public boolean isApplicationExternalIdentifierEnabled(final @PathVariable("identifier") String identifier) {
        SanityChecker.checkSecureParameter(identifier);
        LOGGER.debug("isApplicationExternalIdentifierEnabled");
        return applicationService.isApplicationExternalIdentifierEnabled(identifier);
    }

    @PutMapping(CommonConstants.PATH_ID)
    public ApplicationDto update(final @PathVariable("id") String id, final @Valid @RequestBody ApplicationDto dto) {
        throw new UnsupportedOperationException("update not implemented");
    }

    @PatchMapping(CommonConstants.PATH_ID)
    public ApplicationDto patch(
        final @PathVariable("id") String id,
        @RequestBody final Map<String, Object> partialDto
    ) {
        throw new UnsupportedOperationException("patch not implemented");
    }
}
