/*
 * Decompiled with CFR 0.152.
 */
package fr.gouv.vitam.storage.offers.tape.impl.readwrite;

import com.google.common.util.concurrent.Uninterruptibles;
import fr.gouv.vitam.common.ParametersChecker;
import fr.gouv.vitam.common.guid.GUIDFactory;
import fr.gouv.vitam.common.json.JsonHandler;
import fr.gouv.vitam.common.logging.VitamLogger;
import fr.gouv.vitam.common.logging.VitamLoggerFactory;
import fr.gouv.vitam.storage.engine.common.model.TapeCatalog;
import fr.gouv.vitam.storage.engine.common.model.TapeCatalogLabel;
import fr.gouv.vitam.storage.engine.common.model.TapeLocation;
import fr.gouv.vitam.storage.engine.common.model.TapeLocationType;
import fr.gouv.vitam.storage.engine.common.model.TapeState;
import fr.gouv.vitam.storage.offers.tape.dto.TapeDriveSpec;
import fr.gouv.vitam.storage.offers.tape.dto.TapeDriveStatus;
import fr.gouv.vitam.storage.offers.tape.exception.ReadWriteErrorCode;
import fr.gouv.vitam.storage.offers.tape.exception.ReadWriteException;
import fr.gouv.vitam.storage.offers.tape.exception.TapeCommandException;
import fr.gouv.vitam.storage.offers.tape.spec.TapeDriveService;
import fr.gouv.vitam.storage.offers.tape.spec.TapeLibraryService;
import fr.gouv.vitam.storage.offers.tape.spec.TapeLoadUnloadService;
import fr.gouv.vitam.storage.offers.tape.spec.TapeRobotPool;
import fr.gouv.vitam.storage.offers.tape.spec.TapeRobotService;
import java.io.File;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import org.apache.commons.io.FileUtils;

public class TapeLibraryServiceImpl
implements TapeLibraryService {
    private static final VitamLogger LOGGER = VitamLoggerFactory.getInstance(TapeLibraryServiceImpl.class);
    private static final long SLEEP_TIME = 20L;
    private static final long MB_TO_BYTES = 1000000L;
    private final TapeDriveService tapeDriveService;
    private final TapeRobotPool tapeRobotPool;
    private final int fullCartridgeDetectionThresholdInMB;
    public final String MSG_PREFIX;

    public TapeLibraryServiceImpl(TapeDriveService tapeDriveService, TapeRobotPool tapeRobotPool, int fullCartridgeDetectionThresholdInMB) {
        this.tapeDriveService = tapeDriveService;
        this.tapeRobotPool = tapeRobotPool;
        this.fullCartridgeDetectionThresholdInMB = fullCartridgeDetectionThresholdInMB;
        this.MSG_PREFIX = String.format("[Library] : %s, [Drive] : %s, ", tapeRobotPool.getLibraryIdentifier(), tapeDriveService.getTapeDriveConf().getIndex());
    }

    @Override
    public void goToPosition(TapeCatalog tape, Integer position, ReadWriteErrorCode readWriteErrorCode) throws ReadWriteException {
        if (position == 0) {
            this.rewindTape(tape);
            return;
        }
        int offset = position - tape.getCurrentPosition();
        if (offset == 0) {
            LOGGER.debug("No need to move (current position=" + tape.getCurrentPosition() + ", target position=" + position + ")");
            return;
        }
        try {
            this.tapeDriveService.getDriveCommandService().move(Math.abs(offset), offset < 0);
        }
        catch (TapeCommandException e) {
            throw new ReadWriteException(this.MSG_PREFIX + " [Tape] : " + tape.getCode() + " Action : FSF goto position Error , Entity: " + JsonHandler.unprettyPrint((Object)e.getDetails()), readWriteErrorCode, e);
        }
        tape.setCurrentPosition(position);
    }

    @Override
    public void rewindTape(TapeCatalog tape) throws ReadWriteException {
        try {
            this.tapeDriveService.getDriveCommandService().rewind();
        }
        catch (TapeCommandException e) {
            throw new ReadWriteException(this.MSG_PREFIX + " [Tape] : " + tape.getCode() + " Action : Rewind Tape, Entity: " + JsonHandler.unprettyPrint((Object)e.getDetails()), ReadWriteErrorCode.KO_ON_REWIND_TAPE, e);
        }
        tape.setCurrentPosition(Integer.valueOf(0));
    }

    @Override
    public void write(String filePath, long writtenBytes, TapeCatalog tape) throws ReadWriteException {
        if (tape == null) {
            throw new ReadWriteException(this.MSG_PREFIX + ", Error: can't write, current tape is null.", ReadWriteErrorCode.NULL_CURRENT_TAPE);
        }
        if (tape.getFileCount() < tape.getCurrentPosition()) {
            throw new ReadWriteException(this.MSG_PREFIX + ", Error: current position must be <= to fileCount.", ReadWriteErrorCode.KO_TAPE_CURRENT_POSITION_GREATER_THAN_FILE_COUNT);
        }
        try {
            this.goToPosition(tape, tape.getFileCount(), ReadWriteErrorCode.KO_ON_GOTO_FILE_COUNT);
            try {
                this.tapeDriveService.getReadWriteService().writeToTape(filePath);
            }
            catch (TapeCommandException e) {
                LOGGER.error(this.MSG_PREFIX + " [Tape] : " + tape.getCode() + " Action : Write, Entity: " + JsonHandler.unprettyPrint((Object)e.getDetails()), (Throwable)e);
                TapeDriveSpec status = this.getDriveStatus(ReadWriteErrorCode.KO_DRIVE_STATUS_KO_AFTER_WRITE_ERROR);
                String cartridgeType = status.getCartridge();
                long tapeOccupation = tape.getWrittenBytes() + writtenBytes;
                if (tapeOccupation >= (long)this.fullCartridgeDetectionThresholdInMB * 1000000L) {
                    throw new ReadWriteException(this.MSG_PREFIX + " [Tape] : " + tape.getCode() + " Action : Write, Drive Status: " + JsonHandler.unprettyPrint((Object)status) + ", Error: End Of Tape, Tape Occupation: " + tapeOccupation + " (bytes), Cartridge type: " + cartridgeType + "]", ReadWriteErrorCode.KO_ON_END_OF_TAPE, e);
                }
                tape.setTapeState(TapeState.CONFLICT);
                throw new ReadWriteException(this.MSG_PREFIX + " [Tape] : " + tape.getCode() + " Action : Write, Drive Status: " + JsonHandler.unprettyPrint((Object)status) + ", Error: Tape is CORRUPTED, Cartridge type: '" + cartridgeType + "', Tape space usage: " + tapeOccupation + " (bytes), Full tape threshold: " + (long)this.fullCartridgeDetectionThresholdInMB * 1000000L + " (bytes)", ReadWriteErrorCode.KO_ON_WRITE_TO_TAPE, e);
            }
            tape.setFileCount(Integer.valueOf(tape.getFileCount() + 1));
            tape.setCurrentPosition(tape.getFileCount());
            tape.setWrittenBytes(Long.valueOf(tape.getWrittenBytes() + writtenBytes));
            tape.setTapeState(TapeState.OPEN);
        }
        catch (ReadWriteException e) {
            throw e;
        }
        catch (Exception e) {
            throw new ReadWriteException(this.MSG_PREFIX + " [Tape] : " + tape.getCode(), e);
        }
    }

    @Override
    public void read(TapeCatalog tape, Integer position, String outputPath) throws ReadWriteException {
        this.goToPosition(tape, position, ReadWriteErrorCode.KO_ON_GO_TO_POSITION);
        try {
            this.tapeDriveService.getReadWriteService().readFromTape(outputPath);
            tape.setCurrentPosition(Integer.valueOf(tape.getCurrentPosition() + 1));
        }
        catch (TapeCommandException e) {
            throw new ReadWriteException(this.MSG_PREFIX + " [Tape] : " + tape.getCode() + " Action : Write, Entity: " + JsonHandler.unprettyPrint((Object)e.getDetails()), ReadWriteErrorCode.KO_ON_READ_FROM_TAPE, e);
        }
    }

    @Override
    public void loadTape(TapeCatalog tape) throws ReadWriteException {
        ParametersChecker.checkParameter((String)(this.MSG_PREFIX + ", Error: tape to load is null. please get markReady tape from catalog"), (Object[])new Object[]{tape});
        if (null == tape.getCurrentLocation()) {
            throw new ReadWriteException(this.MSG_PREFIX + " [Tape] : " + tape.getCode() + ", Error: tape current location is null. please update catalog", ReadWriteErrorCode.TAPE_LOCATION_CONFLICT_ON_LOAD);
        }
        Integer driveIndex = this.tapeDriveService.getTapeDriveConf().getIndex();
        Integer slotIndex = tape.getCurrentLocation().getIndex();
        this.loadTapeFromSlotIntoDrive(tape, driveIndex, slotIndex);
        this.rewindTape(tape);
    }

    private void loadTapeFromSlotIntoDrive(TapeCatalog tape, Integer driveIndex, Integer slotIndex) throws ReadWriteException {
        try {
            TapeRobotService tapeRobotService = this.tapeRobotPool.checkoutRobotService();
            try {
                TapeLoadUnloadService loadUnloadService = tapeRobotService.getLoadUnloadService();
                loadUnloadService.loadTape(slotIndex, driveIndex);
                tape.setPreviousLocation(tape.getCurrentLocation());
                tape.setCurrentLocation(new TapeLocation(this.getDriveIndex(), TapeLocationType.DRIVE));
            }
            catch (TapeCommandException e) {
                throw new ReadWriteException(this.MSG_PREFIX + " [Tape] : " + tape.getCode() + ", Action : load, Entity: " + JsonHandler.unprettyPrint((Object)e.getDetails()), ReadWriteErrorCode.KO_ON_LOAD_TAPE, e);
            }
            finally {
                this.tapeRobotPool.pushRobotService(tapeRobotService);
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new ReadWriteException(this.MSG_PREFIX + ", Error: ", e);
        }
    }

    @Override
    public void unloadTape(TapeCatalog tape) throws ReadWriteException {
        ParametersChecker.checkParameter((String)(this.MSG_PREFIX + ", Error: tape to unload is null."), (Object[])new Object[]{tape});
        Integer driveIndex = this.tapeDriveService.getTapeDriveConf().getIndex();
        Integer slotIndex = null;
        if (null != tape.getPreviousLocation()) {
            switch (tape.getPreviousLocation().getLocationType()) {
                case SLOT: 
                case IMPORTEXPORT: {
                    slotIndex = tape.getPreviousLocation().getIndex();
                    break;
                }
                case DRIVE: 
                case OUTSIDE: {
                    throw new ReadWriteException(this.MSG_PREFIX + " [Tape] : " + tape.getCode() + ", Error: previous location should no be in drive", ReadWriteErrorCode.TAPE_LOCATION_CONFLICT_ON_UNLOAD);
                }
                default: {
                    throw new IllegalArgumentException(this.MSG_PREFIX + " [Tape] : " + tape.getCode() + ", Error: location type not implemented");
                }
            }
        }
        if (null == slotIndex) {
            throw new ReadWriteException(this.MSG_PREFIX + " [Tape] : " + tape.getCode() + ", Error : no empty slot found => cannot unload tape", ReadWriteErrorCode.NO_EMPTY_SLOT_FOUND);
        }
        this.ejectTapeFromDrive(tape);
        this.unloadTapeFromDriveToSlot(tape, driveIndex, slotIndex);
    }

    private void ejectTapeFromDrive(TapeCatalog tape) throws ReadWriteException {
        try {
            this.tapeDriveService.getDriveCommandService().eject();
        }
        catch (TapeCommandException e) {
            throw new ReadWriteException(this.MSG_PREFIX + " [Tape] : " + tape.getCode() + " Action : Eject tape with forced rewind, Error: Could not rewind or unload tape, Entity: " + JsonHandler.unprettyPrint((Object)e.getDetails()), ReadWriteErrorCode.KO_REWIND_BEFORE_UNLOAD_TAPE, e);
        }
        tape.setCurrentPosition(Integer.valueOf(0));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void unloadTapeFromDriveToSlot(TapeCatalog tape, Integer driveIndex, Integer slotIndex) throws ReadWriteException {
        try {
            TapeRobotService tapeRobotService = this.tapeRobotPool.checkoutRobotService();
            try {
                TapeLoadUnloadService loadUnloadService = tapeRobotService.getLoadUnloadService();
                try {
                    loadUnloadService.unloadTape(slotIndex, driveIndex);
                }
                catch (TapeCommandException e) {
                    throw new ReadWriteException(this.MSG_PREFIX + " [Tape] : " + tape.getCode() + ", Action : unload, Entity: " + JsonHandler.unprettyPrint((Object)e.getDetails()), ReadWriteErrorCode.KO_ON_UNLOAD_TAPE, e);
                }
                tape.setCurrentLocation(tape.getPreviousLocation());
            }
            finally {
                this.tapeRobotPool.pushRobotService(tapeRobotService);
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new ReadWriteException(this.MSG_PREFIX + " [Tape] : " + tape.getCode(), e);
        }
    }

    @Override
    public TapeDriveSpec getDriveStatus(ReadWriteErrorCode readWriteErrorCode) throws ReadWriteException {
        int retry = 3;
        while (true) {
            try {
                return this.tapeDriveService.getDriveCommandService().status();
            }
            catch (TapeCommandException e) {
                if (--retry == 0) {
                    throw new ReadWriteException(this.MSG_PREFIX + " [Tape] :  Action : drive status, Entity: " + JsonHandler.unprettyPrint((Object)e.getDetails()), readWriteErrorCode);
                }
                LOGGER.error(this.MSG_PREFIX + " [Tape] :  Action : drive status, Entity: " + JsonHandler.unprettyPrint((Object)e.getDetails()), (Object)ReadWriteErrorCode.KO_ON_STATUS, (Object)e);
                Uninterruptibles.sleepUninterruptibly((long)20L, (TimeUnit)TimeUnit.MILLISECONDS);
                continue;
            }
            break;
        }
    }

    @Override
    public Integer getDriveIndex() {
        return this.tapeDriveService.getTapeDriveConf().getIndex();
    }

    @Override
    public String getLibraryIdentifier() {
        return this.tapeRobotPool.getLibraryIdentifier();
    }

    @Override
    public String getTmpOutputDirectory() {
        return this.tapeDriveService.getReadWriteService().getTmpOutputStorageFolder();
    }

    @Override
    public void ensureTapeIsEmpty(TapeCatalog tape, boolean forceOverrideNonEmptyCartridges) throws ReadWriteException {
        block5: {
            try {
                LOGGER.debug(this.MSG_PREFIX + " [Tape] : " + tape.getCode() + ", Action: try read from empty tape (expected to fail)");
                this.tapeDriveService.getDriveCommandService().move(1, false);
                if (forceOverrideNonEmptyCartridges) {
                    LOGGER.warn("OVERRIDING NON EMPTY CARTRIDGE " + tape.getCode());
                    try {
                        this.tapeDriveService.getDriveCommandService().rewind();
                        break block5;
                    }
                    catch (TapeCommandException e) {
                        throw new ReadWriteException(this.MSG_PREFIX + " [Tape] : " + tape.getCode() + " Action : Force override non empty tape, Error: Could not rewind for force empty cartridge overriding, Entity: " + JsonHandler.unprettyPrint((Object)e.getDetails()), ReadWriteErrorCode.KO_REWIND_BEFORE_FORCE_OVERRIDE_NON_EMPTY_TAPE, e);
                    }
                }
                tape.setCurrentPosition(Integer.valueOf(tape.getCurrentPosition() + 1));
                throw new ReadWriteException(this.MSG_PREFIX + " [Tape] : " + tape.getCode() + " Action : Is Tape Empty, Error: Tape not empty but tape catalog is empty", ReadWriteErrorCode.KO_LABEL_DISCORDING_NOT_EMPTY_TAPE);
            }
            catch (TapeCommandException e) {
                LOGGER.debug(this.MSG_PREFIX + " [Tape] : " + tape.getCode() + ", Action: trying read from empty tape failed successfully", (Throwable)e);
            }
        }
        TapeDriveSpec driveStatus = this.getDriveStatus(ReadWriteErrorCode.KO_ON_STATUS);
        tape.setType(driveStatus.getCartridge());
        tape.setWorm(driveStatus.getDriveStatuses().contains((Object)TapeDriveStatus.WR_PROT));
    }

    /*
     * Loose catch block
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public void checkNonEmptyTapeLabel(TapeCatalog tape) throws ReadWriteException {
        File labelFile = null;
        try {
            labelFile = new File(this.getTmpOutputDirectory(), GUIDFactory.newGUID().getId());
            try {
                this.tapeDriveService.getReadWriteService().readFromTape(labelFile.getAbsolutePath());
            }
            catch (TapeCommandException e) {
                throw new ReadWriteException(this.MSG_PREFIX + " [Tape] : " + tape.getCode() + " Action : Read from tape, Entity: " + JsonHandler.unprettyPrint((Object)e.getDetails()), ReadWriteErrorCode.KO_ON_READ_LABEL, e);
            }
            TapeCatalogLabel tapeLabel = (TapeCatalogLabel)JsonHandler.getFromFile((File)labelFile, TapeCatalogLabel.class);
            TapeCatalogLabel tapeCatalogLabel = tape.getLabel();
            if (!Objects.equals(tapeLabel.getId(), tapeCatalogLabel.getId())) {
                throw new ReadWriteException(this.MSG_PREFIX + " [Tape] : " + tape.getCode() + " Action : Check tape label, Expected label: " + tapeCatalogLabel.getId() + ", read label: " + tapeLabel.getId(), ReadWriteErrorCode.KO_LABEL_DISCORDING);
            }
            tape.setCurrentPosition(Integer.valueOf(1));
        }
        catch (ReadWriteException e) {
            try {
                throw e;
                catch (Exception e2) {
                    throw new ReadWriteException(this.MSG_PREFIX + " [Tape] : " + tape.getCode(), e2);
                }
            }
            catch (Throwable throwable) {
                FileUtils.deleteQuietly(labelFile);
                throw throwable;
            }
        }
        FileUtils.deleteQuietly((File)labelFile);
    }
}

