/*
 * Decompiled with CFR 0.152.
 */
package fr.gouv.vitam.storage.engine.server.offerdiff;

import fr.gouv.vitam.common.LocalDateUtil;
import fr.gouv.vitam.common.VitamConfiguration;
import fr.gouv.vitam.common.collection.CloseableIterator;
import fr.gouv.vitam.common.json.JsonHandler;
import fr.gouv.vitam.common.logging.VitamLogger;
import fr.gouv.vitam.common.logging.VitamLoggerFactory;
import fr.gouv.vitam.common.model.StatusCode;
import fr.gouv.vitam.common.model.storage.ObjectEntry;
import fr.gouv.vitam.common.model.storage.ObjectEntryReader;
import fr.gouv.vitam.common.model.storage.ObjectEntryWriter;
import fr.gouv.vitam.common.thread.VitamThreadFactory;
import fr.gouv.vitam.common.thread.VitamThreadUtils;
import fr.gouv.vitam.storage.engine.common.exception.StorageException;
import fr.gouv.vitam.storage.engine.common.model.DataCategory;
import fr.gouv.vitam.storage.engine.common.referential.model.OfferReference;
import fr.gouv.vitam.storage.engine.server.distribution.StorageDistribution;
import fr.gouv.vitam.storage.engine.server.offerdiff.OfferDiffStatus;
import fr.gouv.vitam.storage.engine.server.offerdiff.ReportWriter;
import fr.gouv.vitam.storage.engine.server.offerdiff.sort.LargeFileSorter;
import fr.gouv.vitam.storage.engine.server.offerdiff.sort.ObjectEntryLargeFileReader;
import fr.gouv.vitam.storage.engine.server.offerdiff.sort.ObjectEntryLargeFileWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.stream.Collectors;
import org.apache.commons.collections4.iterators.PeekingIterator;
import org.apache.commons.io.FileUtils;

public class OfferDiffProcess {
    private static final VitamLogger LOGGER = VitamLoggerFactory.getInstance(OfferDiffProcess.class);
    private final StorageDistribution distribution;
    private final String offer1;
    private final String offer2;
    private final DataCategory dataCategory;
    private OfferDiffStatus offerDiffStatus;

    public OfferDiffProcess(StorageDistribution distribution, String offer1, String offer2, DataCategory dataCategory) {
        this.distribution = distribution;
        this.offer1 = offer1;
        this.offer2 = offer2;
        this.dataCategory = dataCategory;
        this.offerDiffStatus = new OfferDiffStatus().setRequestId(VitamThreadUtils.getVitamSession().getRequestId()).setTenantId(VitamThreadUtils.getVitamSession().getTenantId()).setStatusCode(StatusCode.UNKNOWN).setOffer1(offer1).setOffer2(offer2).setContainer(dataCategory.getCollectionName()).setStartDate(this.getCurrentDate()).setEndDate(null);
    }

    public void run() {
        File diffOperationTempDir = null;
        try {
            LOGGER.info(String.format("[OfferDiff] Start offer diff process for offers '{%s}' and '{%s}' for category {%s}.", this.offer1, this.offer2, this.dataCategory));
            Set offerIds = this.distribution.getStrategies().values().stream().flatMap(s -> s.getOffers().stream()).map(OfferReference::getId).collect(Collectors.toSet());
            if (!offerIds.contains(this.offer1)) {
                throw new IllegalArgumentException("Invalid offer1 : '" + this.offer1 + "'");
            }
            if (!offerIds.contains(this.offer2)) {
                throw new IllegalArgumentException("Invalid offer2 : '" + this.offer2 + "'");
            }
            diffOperationTempDir = this.createProcessTempStorageDir();
            this.process(this.offer1, this.offer2, this.dataCategory, diffOperationTempDir);
            LOGGER.info("[OfferDiff] Offer diff completed successfully");
            this.offerDiffStatus.setStatusCode(this.offerDiffStatus.getErrorCount() == 0L ? StatusCode.OK : StatusCode.WARNING);
        }
        catch (Exception e) {
            LOGGER.error("[OfferDiff]: An exception has been thrown during offer diff process.", (Throwable)e);
            this.offerDiffStatus.setStatusCode(StatusCode.KO);
            this.cleanupTempDirFilesQuietly(diffOperationTempDir);
        }
        finally {
            this.offerDiffStatus.setEndDate(this.getCurrentDate());
            LOGGER.info("[OfferDiff] Offer diff result:\n" + JsonHandler.prettyPrint((Object)this.offerDiffStatus));
        }
    }

    private void process(String offer1, String offer2, DataCategory dataCategory, File diffOperationTempDir) throws StorageException {
        LOGGER.info("[OfferDiff] Listing offer files...");
        ExecutorService executor = Executors.newFixedThreadPool(2, (ThreadFactory)VitamThreadFactory.getInstance());
        int tenant = VitamThreadUtils.getVitamSession().getTenantId();
        String requestId = VitamThreadUtils.getVitamSession().getRequestId();
        CompletableFuture<File> listOffers1CompletableFuture = CompletableFuture.supplyAsync(() -> this.listOfferObjects(offer1, dataCategory, diffOperationTempDir, tenant, requestId), executor);
        CompletableFuture<File> listOffers2CompletableFuture = CompletableFuture.supplyAsync(() -> this.listOfferObjects(offer2, dataCategory, diffOperationTempDir, tenant, requestId), executor);
        Optional<File> offerListing1 = this.await(listOffers1CompletableFuture, offer1);
        Optional<File> offerListing2 = this.await(listOffers2CompletableFuture, offer2);
        executor.shutdown();
        if (offerListing1.isEmpty() || offerListing2.isEmpty()) {
            throw new StorageException("One or more offer listing failed. Aborting");
        }
        LOGGER.info("[OfferDiff] Sorting offer files...");
        File sortedOfferListing1 = this.sort(offer1, offerListing1.get(), diffOperationTempDir);
        File sortedOfferListing2 = this.sort(offer2, offerListing2.get(), diffOperationTempDir);
        LOGGER.info("[OfferDiff] Comparing offer files...");
        try {
            File reportFile = this.createTempFile(diffOperationTempDir, "_" + this.offerDiffStatus.getRequestId() + ".jsonl");
            try (ReportWriter reportWriter = new ReportWriter(reportFile);){
                this.compareOfferListings(offer1, offer2, sortedOfferListing1, sortedOfferListing2, reportWriter);
                this.offerDiffStatus.setReportFileName(reportFile.getAbsoluteFile().toString());
                this.offerDiffStatus.setTotalObjectCount(reportWriter.getTotalObjectCount());
                this.offerDiffStatus.setErrorCount(reportWriter.getErrorCount());
                this.cleanupFiles(sortedOfferListing1, sortedOfferListing2);
            }
        }
        catch (IOException e) {
            throw new StorageException("Could not compare sorted offer listings", (Throwable)e);
        }
        LOGGER.info("[OfferDiff] Comparing offer files done successfully !");
    }

    private File listOfferObjects(String offerId, DataCategory dataCategory, File diffOperationTempDir, Integer tenant, String requestId) {
        File file;
        block21: {
            Thread.currentThread().setName("OfferListing-" + offerId);
            VitamThreadUtils.getVitamSession().setTenantId(tenant);
            VitamThreadUtils.getVitamSession().setRequestId(requestId);
            CloseableIterator<ObjectEntry> objectEntryIterator = this.distribution.listContainerObjectsForOffer(dataCategory, offerId, true);
            try {
                File tempFile = this.createTempFile(diffOperationTempDir);
                try (FileOutputStream fos = new FileOutputStream(tempFile);
                     ObjectEntryWriter objectEntryWriter = new ObjectEntryWriter((OutputStream)fos);){
                    int i = 0;
                    while (objectEntryIterator.hasNext()) {
                        objectEntryWriter.write((ObjectEntry)objectEntryIterator.next());
                        if (i > 0 && i % 10000 == 0) {
                            LOGGER.info("[OfferDiff] " + i + "/? files listed from " + offerId + "/" + String.valueOf(dataCategory));
                        }
                        ++i;
                    }
                    objectEntryWriter.writeEof();
                }
                file = tempFile;
                if (objectEntryIterator == null) break block21;
            }
            catch (Throwable throwable) {
                try {
                    if (objectEntryIterator != null) {
                        try {
                            objectEntryIterator.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (StorageException | IOException e) {
                    throw new RuntimeException("Could not list offer objects. OfferId: '" + offerId + "', dataCategory: " + String.valueOf(dataCategory));
                }
            }
            objectEntryIterator.close();
        }
        return file;
    }

    private Optional<File> await(CompletableFuture<File> completableFuture, String offerId) {
        try {
            File offerListingFile = completableFuture.get();
            LOGGER.info("Offer listing succeeded for offer " + offerId);
            return Optional.of(offerListingFile);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            LOGGER.error("Interrupted thread while listing offer objects for " + offerId, (Throwable)e);
            return Optional.empty();
        }
        catch (ExecutionException e) {
            LOGGER.error("An error occurred during offer listing for " + offerId, (Throwable)e);
            return Optional.empty();
        }
    }

    private File sort(String offerId, File offerListing, File diffOperationTempDir) throws StorageException {
        LOGGER.info("[OfferDiff] Sorting offer file " + offerId);
        try {
            LargeFileSorter<ObjectEntry> objectEntryLargeFileSorter = new LargeFileSorter<ObjectEntry>(ObjectEntryLargeFileReader::new, ObjectEntryLargeFileWriter::new, Comparator.comparing(ObjectEntry::getObjectId), () -> {
                try {
                    return this.createTempFile(diffOperationTempDir);
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            });
            File result = objectEntryLargeFileSorter.sortLargeFile(offerListing);
            LOGGER.info("[OfferDiff] Offer listing file " + offerId + " sorted successfully !");
            File file = result;
            return file;
        }
        catch (IOException e) {
            throw new StorageException("Cloud not sort offer listing file for offer " + offerId, (Throwable)e);
        }
        finally {
            FileUtils.deleteQuietly((File)offerListing);
        }
    }

    private void compareOfferListings(String offer1, String offer2, File sortedOfferListing1, File sortedOfferListing2, ReportWriter reportWriter) throws IOException {
        try (FileInputStream inputStream1 = new FileInputStream(sortedOfferListing1);
             ObjectEntryReader objectEntryReader1 = new ObjectEntryReader((InputStream)inputStream1);
             FileInputStream inputStream2 = new FileInputStream(sortedOfferListing2);
             ObjectEntryReader objectEntryReader2 = new ObjectEntryReader((InputStream)inputStream2);){
            PeekingIterator peekingIterator1 = new PeekingIterator((Iterator)objectEntryReader1);
            PeekingIterator peekingIterator2 = new PeekingIterator((Iterator)objectEntryReader2);
            while (peekingIterator1.hasNext() || peekingIterator2.hasNext()) {
                if (peekingIterator1.hasNext() && peekingIterator2.hasNext()) {
                    String objectId2;
                    String objectId1 = ((ObjectEntry)peekingIterator1.peek()).getObjectId();
                    if (objectId1.equals(objectId2 = ((ObjectEntry)peekingIterator2.peek()).getObjectId())) {
                        this.reportDifference((ObjectEntry)peekingIterator1.next(), (ObjectEntry)peekingIterator2.next(), offer1, offer2, reportWriter);
                        continue;
                    }
                    if (objectId1.compareTo(objectId2) < 0) {
                        this.reportDifference((ObjectEntry)peekingIterator1.next(), null, offer1, offer2, reportWriter);
                        continue;
                    }
                    this.reportDifference(null, (ObjectEntry)peekingIterator2.next(), offer1, offer2, reportWriter);
                    continue;
                }
                if (peekingIterator1.hasNext()) {
                    this.reportDifference((ObjectEntry)peekingIterator1.next(), null, offer1, offer2, reportWriter);
                    continue;
                }
                this.reportDifference(null, (ObjectEntry)peekingIterator2.next(), offer1, offer2, reportWriter);
            }
        }
    }

    private void reportDifference(ObjectEntry objectEntry1, ObjectEntry objectEntry2, String offer1, String offer2, ReportWriter reportWriter) throws IOException {
        if (objectEntry1 != null && objectEntry2 != null) {
            if (objectEntry1.getSize() == objectEntry2.getSize()) {
                LOGGER.debug("File " + objectEntry1.getObjectId() + " matches. Size: " + objectEntry1.getSize());
                reportWriter.reportMatchingObject(objectEntry1.getObjectId());
            } else {
                LOGGER.warn("File " + objectEntry1.getObjectId() + " found on both offers, but size mismatches. " + offer1 + ": " + objectEntry1.getSize() + "bytes , " + offer2 + ": " + objectEntry2.getSize() + " bytes");
                reportWriter.reportObjectMismatch(objectEntry1.getObjectId(), objectEntry1.getSize(), objectEntry2.getSize());
            }
        } else if (objectEntry1 != null) {
            LOGGER.warn("File " + objectEntry1.getObjectId() + " found on offer " + offer1 + " (" + objectEntry1.getSize() + "bytes), but is missing from offer " + offer2);
            reportWriter.reportObjectMismatch(objectEntry1.getObjectId(), objectEntry1.getSize(), null);
        } else {
            LOGGER.warn("File " + objectEntry2.getObjectId() + " found on offer " + offer2 + " (" + objectEntry2.getSize() + "bytes), but is missing from offer " + offer1);
            reportWriter.reportObjectMismatch(objectEntry2.getObjectId(), null, objectEntry2.getSize());
        }
    }

    private void cleanupFiles(File ... files) {
        for (File file : files) {
            FileUtils.deleteQuietly((File)file);
        }
    }

    public boolean isRunning() {
        return this.offerDiffStatus.getEndDate() == null;
    }

    public OfferDiffStatus getOfferDiffStatus() {
        return this.offerDiffStatus;
    }

    private String getCurrentDate() {
        return LocalDateUtil.nowFormatted();
    }

    private File createTempFile(File diffOperationTempDir) throws IOException {
        return this.createTempFile(diffOperationTempDir, null);
    }

    private File createTempFile(File diffOperationTempDir, String suffix) throws IOException {
        return File.createTempFile("offer_diff_", suffix, diffOperationTempDir);
    }

    private void cleanupTempDirFilesQuietly(File diffOperationTempDir) {
        try {
            if (diffOperationTempDir == null) {
                return;
            }
            FileUtils.forceDelete((File)diffOperationTempDir);
        }
        catch (IOException ex) {
            LOGGER.error("Could not clean temp folder");
        }
    }

    private File createProcessTempStorageDir() throws IOException {
        File tempDir = new File(VitamConfiguration.getVitamTmpFolder());
        File diffOperationTempDir = new File(tempDir, "offer_diff_" + VitamThreadUtils.getVitamSession().getRequestId());
        FileUtils.forceMkdir((File)diffOperationTempDir);
        return diffOperationTempDir;
    }
}

