/*
 * Decompiled with CFR 0.152.
 */
package fr.gouv.vitam.common.storage.filesystem.v2;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Stopwatch;
import fr.gouv.vitam.common.ParametersChecker;
import fr.gouv.vitam.common.VitamConfiguration;
import fr.gouv.vitam.common.digest.DigestType;
import fr.gouv.vitam.common.logging.VitamLogger;
import fr.gouv.vitam.common.logging.VitamLoggerFactory;
import fr.gouv.vitam.common.model.MetadatasObject;
import fr.gouv.vitam.common.model.storage.ObjectEntry;
import fr.gouv.vitam.common.performance.PerformanceLogger;
import fr.gouv.vitam.common.storage.ContainerInformation;
import fr.gouv.vitam.common.storage.StorageConfiguration;
import fr.gouv.vitam.common.storage.cas.container.api.ContentAddressableStorageAbstract;
import fr.gouv.vitam.common.storage.cas.container.api.MetadatasStorageObject;
import fr.gouv.vitam.common.storage.cas.container.api.ObjectContent;
import fr.gouv.vitam.common.storage.cas.container.api.ObjectListingListener;
import fr.gouv.vitam.common.storage.constants.ErrorMessage;
import fr.gouv.vitam.common.storage.constants.ExtendedAttributes;
import fr.gouv.vitam.common.storage.filesystem.v2.HashFileSystemHelper;
import fr.gouv.vitam.workspace.api.exception.ContentAddressableStorageAlreadyExistException;
import fr.gouv.vitam.workspace.api.exception.ContentAddressableStorageDigestMismatchException;
import fr.gouv.vitam.workspace.api.exception.ContentAddressableStorageException;
import fr.gouv.vitam.workspace.api.exception.ContentAddressableStorageNotFoundException;
import fr.gouv.vitam.workspace.api.exception.ContentAddressableStorageServerException;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileSystemException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.UserDefinedFileAttributeView;
import java.util.concurrent.TimeUnit;

public class HashFileSystem
extends ContentAddressableStorageAbstract {
    private static final VitamLogger LOGGER = VitamLoggerFactory.getInstance(HashFileSystem.class);
    private static final String ERROR_MSG_NOT_SUPPORTED = "Extended attribute not supported. You should consider to use XFS filesystem.";
    private final HashFileSystemHelper fsHelper;

    public HashFileSystem(StorageConfiguration configuration) {
        super(configuration);
        ParametersChecker.checkParameter((String)"Storage configuration can't be null", (Object[])new Object[]{configuration});
        ParametersChecker.checkParameter((String)"StoragePath can't be null", (String[])new String[]{configuration.getStoragePath()});
        String storagePath = configuration.getStoragePath();
        this.fsHelper = new HashFileSystemHelper(storagePath);
        File f = new File(storagePath);
        if (!f.exists()) {
            throw new IllegalArgumentException("The storage path doesn't exist");
        }
        if (!f.isDirectory()) {
            throw new IllegalArgumentException("The storage path is not a directory");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void createContainer(String containerName) throws ContentAddressableStorageServerException {
        Class<HashFileSystem> clazz = HashFileSystem.class;
        synchronized (HashFileSystem.class) {
            ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_NAME_IS_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName});
            if (this.isExistingContainer(containerName)) {
                LOGGER.warn("Container " + containerName + " already exists");
                // ** MonitorExit[var2_2] (shouldn't be in output)
                return;
            }
            this.fsHelper.createContainer(containerName);
            // ** MonitorExit[var2_2] (shouldn't be in output)
            return;
        }
    }

    @Override
    public boolean isExistingContainer(String containerName) {
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_NAME_IS_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName});
        if (super.isExistingContainerInCache(containerName)) {
            return true;
        }
        boolean exists = this.fsHelper.isContainer(containerName);
        this.cacheExistsContainer(containerName, exists);
        return exists;
    }

    @Override
    public void writeObject(String containerName, String objectName, InputStream inputStream, DigestType digestType, long size) throws ContentAddressableStorageException {
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_NAME_IS_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName});
        Path filePath = this.fsHelper.getPathObject(containerName, objectName);
        Path parentPath = filePath.getParent();
        this.fsHelper.createDirectories(parentPath);
        try {
            Files.copy(inputStream, filePath, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (FileAlreadyExistsException e) {
            throw new ContentAddressableStorageAlreadyExistException("File " + String.valueOf(filePath) + " already exists", (Throwable)e);
        }
        catch (IOException e) {
            throw new ContentAddressableStorageServerException("I/O Error on writing file " + String.valueOf(filePath), (Throwable)e);
        }
    }

    @Override
    public void checkObjectDigestAndStoreDigest(String containerName, String objectName, String objectDigest, DigestType digestType, long size) throws ContentAddressableStorageException {
        String computedDigest = this.computeObjectDigest(containerName, objectName, digestType);
        if (!objectDigest.equals(computedDigest)) {
            throw new ContentAddressableStorageDigestMismatchException("Illegal state for container " + containerName + " and  object " + objectName + ". Stream digest " + objectDigest + " is not equal to computed digest " + computedDigest);
        }
        this.storeDigest(containerName, objectName, digestType, objectDigest);
    }

    @Override
    public ObjectContent getObject(String containerName, String objectName) throws ContentAddressableStorageException {
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_NAME_IS_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName});
        Path filePath = this.fsHelper.getPathObject(containerName, objectName);
        if (!filePath.toFile().isFile()) {
            throw new ContentAddressableStorageNotFoundException(objectName + " in container " + containerName + " not found");
        }
        try {
            long size = Files.size(filePath);
            InputStream inputStream = Files.newInputStream(filePath, new OpenOption[0]);
            return new ObjectContent(inputStream, size);
        }
        catch (IOException e) {
            throw new ContentAddressableStorageException("I/O error on retrieving object " + objectName + " in the container " + containerName, (Throwable)e);
        }
    }

    @Override
    public void deleteObject(String containerName, String objectName) throws ContentAddressableStorageException {
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_NAME_IS_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName});
        Path filePath = this.fsHelper.getPathObject(containerName, objectName);
        try {
            Files.delete(filePath);
        }
        catch (NoSuchFileException e) {
            throw new ContentAddressableStorageNotFoundException(String.valueOf((Object)ErrorMessage.OBJECT_NOT_FOUND) + objectName, (Throwable)e);
        }
        catch (IOException e) {
            throw new ContentAddressableStorageServerException("I/O error on removing " + String.valueOf(filePath), (Throwable)e);
        }
        try {
            Path containerPath = this.fsHelper.getPathContainer(containerName);
            filePath = filePath.getParent();
            while (!filePath.equals(containerPath)) {
                Files.delete(filePath);
                filePath = filePath.getParent();
            }
        }
        catch (DirectoryNotEmptyException containerPath) {
        }
        catch (IOException e) {
            LOGGER.warn("Impossible to remove directory" + String.valueOf(filePath), (Throwable)e);
        }
    }

    @Override
    public boolean isExistingObject(String containerName, String objectName) throws ContentAddressableStorageServerException {
        Path filePath;
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_NAME_IS_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName});
        try {
            filePath = this.fsHelper.getPathObject(containerName, objectName);
        }
        catch (ContentAddressableStorageNotFoundException e) {
            return false;
        }
        return filePath.toFile().isFile();
    }

    @Override
    public String getObjectDigest(String containerName, String objectName, DigestType algo, boolean noCache) throws ContentAddressableStorageException {
        ParametersChecker.checkParameter((String)ErrorMessage.ALGO_IS_A_MANDATORY_PARAMETER.getMessage(), (Object[])new Object[]{algo});
        String digestFromMD = this.getObjectDigestFromMD(containerName, objectName, algo);
        if (!noCache) {
            if (digestFromMD != null) {
                return digestFromMD;
            }
            LOGGER.warn(String.format("Could not retrieve cached digest for object '%s' in container '%s'. Recomputing digest", objectName, containerName));
        }
        String digest = this.computeObjectDigest(containerName, objectName, algo);
        if (digestFromMD == null || !digestFromMD.equals(digest)) {
            this.storeDigest(containerName, objectName, algo, digest);
        }
        return digest;
    }

    String getObjectDigestFromMD(String containerName, String objectName, DigestType algo) throws ContentAddressableStorageException {
        Stopwatch stopwatch = Stopwatch.createStarted();
        Path filePath = this.fsHelper.getPathObject(containerName, objectName);
        String digestMetadata = null;
        try {
            digestMetadata = this.readExtendedMetadata(filePath, ExtendedAttributes.DIGEST.getKey());
        }
        catch (IOException e) {
            LOGGER.warn("Unable to retrieve DIGEST extended attribute for the object " + objectName + " in the container " + containerName, (Throwable)e);
        }
        String digestFromMD = null;
        if (digestMetadata != null) {
            String[] digestTokens = digestMetadata.split(":");
            try {
                if (digestTokens.length == 2 && DigestType.fromValue((String)digestTokens[0]) == algo) {
                    digestFromMD = digestTokens[1];
                }
            }
            catch (IllegalArgumentException e) {
                LOGGER.warn("DigestAlgorithm in the extended attribute of file " + containerName + "/" + objectName + " is unknown : " + digestTokens[0], (Throwable)e);
            }
        }
        PerformanceLogger.getInstance().log("STP_Offer_" + this.getConfiguration().getProvider(), containerName, "READ_DIGEST_FROM_XATTR", stopwatch.elapsed(TimeUnit.MILLISECONDS));
        return digestFromMD;
    }

    private void storeDigest(String containerName, String objectName, DigestType algo, String digest) throws ContentAddressableStorageNotFoundException, ContentAddressableStorageServerException {
        StringBuilder sb = new StringBuilder(algo.getName()).append(":").append(digest);
        try {
            Stopwatch stopwatch = Stopwatch.createStarted();
            this.writeExtendedMetadata(this.fsHelper.getPathObject(containerName, objectName), ExtendedAttributes.DIGEST.getKey(), sb.toString());
            PerformanceLogger.getInstance().log("STP_Offer_" + this.getConfiguration().getProvider(), containerName, "STORE_DIGEST_TO_XATTR", stopwatch.elapsed(TimeUnit.MILLISECONDS));
        }
        catch (IOException e) {
            LOGGER.warn("Unable to write DIGEST extended attribute for the object " + objectName + " in the container " + containerName, (Throwable)e);
        }
    }

    @Override
    public ContainerInformation getContainerInformation(String containerName) throws ContentAddressableStorageNotFoundException, ContentAddressableStorageServerException {
        if (containerName == null) {
            containerName = "";
        }
        if (!this.isExistingContainer(containerName)) {
            throw new ContentAddressableStorageNotFoundException(String.valueOf((Object)ErrorMessage.CONTAINER_NOT_FOUND) + containerName);
        }
        File containerDir = this.fsHelper.getPathContainer(containerName).toFile();
        long usableSpace = containerDir.getUsableSpace();
        ContainerInformation containerInformation = new ContainerInformation();
        containerInformation.setUsableSpace(usableSpace);
        return containerInformation;
    }

    @Override
    public MetadatasObject getObjectMetadata(String containerName, String objectId, boolean noCache) throws ContentAddressableStorageException {
        try {
            ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_OBJECT_NAMES_ARE_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName, objectId});
            MetadatasStorageObject result = new MetadatasStorageObject();
            Path filePath = this.fsHelper.getPathObject(containerName, objectId);
            File file = filePath.toFile();
            BasicFileAttributes basicAttribs = this.getFileAttributes(file);
            long size = Files.size(Paths.get(file.getPath(), new String[0]));
            result.setObjectName(objectId);
            result.setDigest(this.getObjectDigest(containerName, objectId, VitamConfiguration.getDefaultDigestType(), noCache));
            result.setFileSize(size);
            result.setType(containerName.split("_")[1]);
            result.setLastAccessDate(basicAttribs.lastAccessTime().toString());
            result.setLastModifiedDate(basicAttribs.lastModifiedTime().toString());
            return result;
        }
        catch (FileNotFoundException | NoSuchFileException fnfe) {
            throw new ContentAddressableStorageNotFoundException("Object " + objectId + "for container " + containerName + " is not found", (Throwable)fnfe);
        }
        catch (IOException io) {
            throw new ContentAddressableStorageException("Object " + objectId + "for container " + containerName + " is not accessible", (Throwable)io);
        }
    }

    @Override
    public void listContainer(String containerName, final ObjectListingListener objectListingListener) throws ContentAddressableStorageNotFoundException, ContentAddressableStorageServerException, IOException {
        ParametersChecker.checkParameter((String)ErrorMessage.CONTAINER_NAME_IS_A_MANDATORY_PARAMETER.getMessage(), (String[])new String[]{containerName});
        if (!this.isExistingContainer(containerName)) {
            throw new ContentAddressableStorageNotFoundException(String.valueOf((Object)ErrorMessage.CONTAINER_NOT_FOUND) + containerName);
        }
        Path path = this.fsHelper.getPathContainer(containerName);
        Files.walkFileTree(path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                objectListingListener.handleObjectEntry(new ObjectEntry(file.getFileName().toString(), file.toFile().length()));
                return super.visitFile(file, attrs);
            }
        });
    }

    public void close() {
    }

    private BasicFileAttributes getFileAttributes(File file) throws IOException {
        Path path = Paths.get(file.getPath(), new String[0]);
        BasicFileAttributeView basicView = Files.getFileAttributeView(path, BasicFileAttributeView.class, new LinkOption[0]);
        return basicView.readAttributes();
    }

    private void writeExtendedMetadata(Path p, String name, String value) throws IOException {
        this.writeExtendedMetadata(Files.getFileAttributeView(p, UserDefinedFileAttributeView.class, new LinkOption[0]), name, value);
    }

    private void writeExtendedMetadata(UserDefinedFileAttributeView view, String name, String value) throws IOException {
        try {
            view.write(name, ByteBuffer.wrap(value.getBytes()));
        }
        catch (SecurityException | FileSystemException e) {
            LOGGER.error(ERROR_MSG_NOT_SUPPORTED, (Throwable)e);
            throw new IOException(ERROR_MSG_NOT_SUPPORTED, e);
        }
    }

    @VisibleForTesting
    String readExtendedMetadata(Path p, String name) throws IOException {
        return this.readExtendedMetadata(Files.getFileAttributeView(p, UserDefinedFileAttributeView.class, new LinkOption[0]), name);
    }

    private String readExtendedMetadata(UserDefinedFileAttributeView view, String name) throws IOException {
        try {
            ByteBuffer bb = ByteBuffer.allocate(view.size(name));
            view.read(name, bb);
            bb.flip();
            CharBuffer buffer = Charset.defaultCharset().decode(bb);
            return buffer.toString();
        }
        catch (IllegalArgumentException | FileSystemException e) {
            LOGGER.error(ERROR_MSG_NOT_SUPPORTED, (Throwable)e);
            throw new IOException(ERROR_MSG_NOT_SUPPORTED, e);
        }
    }
}

