/**
 * Copyright French Prime minister Office/SGMAP/DINSIC/Vitam Program (2019-2020)
 * and the signatories of the "VITAM - Accord du Contributeur" agreement.
 *
 * contact@programmevitam.fr
 *
 * This software is a computer program whose purpose is to implement
 * implement a digital archiving front-office system for the secure and
 * efficient high volumetry VITAM solution.
 *
 * This software is governed by the CeCILL-C license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL-C
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided only
 * with a limited warranty  and the software's author,  the holder of the
 * economic rights,  and the successive licensors  have only  limited
 * liability.
 *
 * In this respect, the user's attention is drawn to the risks associated
 * with loading,  using,  modifying and/or developing or reproducing the
 * software by the user in light of its specific status of free software,
 * that may mean  that it is complicated to manipulate,  and  that  also
 * therefore means  that it is reserved for developers  and  experienced
 * professionals having in-depth computer knowledge. Users are therefore
 * encouraged to load and test the software's suitability as regards their
 * requirements in conditions enabling the security of their systems and/or
 * data to be ensured and,  more generally, to use and operate it in the
 * same conditions as regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL-C license and that you accept its terms.
 */
package fr.gouv.vitamui.referential.internal.server.rest;

import com.fasterxml.jackson.databind.JsonNode;
import fr.gouv.vitam.common.client.VitamContext;
import fr.gouv.vitam.common.exception.VitamClientException;
import fr.gouv.vitamui.common.security.SafeFileChecker;
import fr.gouv.vitamui.common.security.SanityChecker;
import fr.gouv.vitamui.commons.api.CommonConstants;
import fr.gouv.vitamui.commons.api.domain.DirectionDto;
import fr.gouv.vitamui.commons.api.domain.PaginatedValuesDto;
import fr.gouv.vitamui.commons.api.logger.VitamUILogger;
import fr.gouv.vitamui.commons.api.logger.VitamUILoggerFactory;
import fr.gouv.vitamui.commons.rest.util.RestUtils;
import fr.gouv.vitamui.iam.security.service.InternalSecurityService;
import fr.gouv.vitamui.referential.common.dto.FileFormatDto;
import fr.gouv.vitamui.referential.common.rest.RestApi;
import fr.gouv.vitamui.referential.internal.server.fileformat.FileFormatInternalService;
import lombok.Getter;
import lombok.Setter;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.InputStreamResource;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;
import javax.ws.rs.Consumes;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Response;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.Map;
import java.util.Optional;

@RestController
@RequestMapping(RestApi.FILE_FORMATS_URL)
@Getter
@Setter
public class FileFormatInternalController {

    private static final VitamUILogger LOGGER = VitamUILoggerFactory.getInstance(FileFormatInternalController.class);

    @Autowired
    private FileFormatInternalService fileFormatInternalService;

    @Autowired
    private InternalSecurityService securityService;

    @GetMapping
    public Collection<FileFormatDto> getAll(@RequestParam final Optional<String> criteria) {
        LOGGER.debug("get all file formats criteria={}", criteria);
        SanityChecker.sanitizeCriteria(criteria);
        final VitamContext vitamContext = securityService.buildVitamContext(securityService.getTenantIdentifier());
        LOGGER.debug("context={}", vitamContext);
        return fileFormatInternalService.getAll(vitamContext);
    }

    @GetMapping(params = { "page", "size" })
    public PaginatedValuesDto<FileFormatDto> getAllPaginated(
        @RequestParam final Integer page,
        @RequestParam final Integer size,
        @RequestParam(required = false) final Optional<String> criteria,
        @RequestParam(required = false) final Optional<String> orderBy,
        @RequestParam(required = false) final Optional<DirectionDto> direction
    ) {
        LOGGER.debug(
            "getPaginateEntities page={}, size={}, criteria={}, orderBy={}, ascendant={}",
            page,
            size,
            criteria,
            orderBy,
            direction
        );
        final VitamContext vitamContext = securityService.buildVitamContext(securityService.getTenantIdentifier());
        return fileFormatInternalService.getAllPaginated(page, size, orderBy, direction, vitamContext, criteria);
    }

    @RequestMapping(value = "/**", method = RequestMethod.GET)
    public Object getByIdOrHistory(HttpServletRequest request)
        throws UnsupportedEncodingException, VitamClientException {
        LOGGER.debug("getByIdOrHistory ");
        String requestURL = request.getRequestURL().toString();
        String path = StringUtils.substringAfter(requestURL, RestApi.FILE_FORMATS_URL + "/");
        if (StringUtils.endsWith(path, "/history")) {
            return findHistoryById(StringUtils.substringBefore(path, "/history"));
        } else {
            return getOne(StringUtils.removeEndIgnoreCase(path, "/"));
        }
    }

    private FileFormatDto getOne(final @PathVariable("identifier") String identifier)
        throws UnsupportedEncodingException {
        LOGGER.debug(
            "get file format identifier={} / {}",
            identifier,
            URLDecoder.decode(identifier, StandardCharsets.UTF_8.toString())
        );
        final VitamContext vitamContext = securityService.buildVitamContext(securityService.getTenantIdentifier());
        return fileFormatInternalService.getOne(
            vitamContext,
            URLDecoder.decode(identifier, StandardCharsets.UTF_8.toString())
        );
    }

    @PostMapping(CommonConstants.PATH_CHECK)
    public ResponseEntity<Void> checkExist(
        @RequestBody FileFormatDto fileFormat,
        @RequestHeader(value = CommonConstants.X_TENANT_ID_HEADER) Integer tenant
    ) {
        LOGGER.debug("check exist file format={}", fileFormat);
        final VitamContext vitamContext = securityService.buildVitamContext(securityService.getTenantIdentifier());
        final boolean exist = fileFormatInternalService.check(vitamContext, fileFormat);
        return RestUtils.buildBooleanResponse(exist);
    }

    @PostMapping
    public FileFormatDto create(
        @Valid @RequestBody FileFormatDto fileFormat,
        @RequestHeader(value = CommonConstants.X_TENANT_ID_HEADER) Integer tenant
    ) {
        LOGGER.debug("create file format={}", fileFormat);
        final VitamContext vitamContext = securityService.buildVitamContext(securityService.getTenantIdentifier());
        return fileFormatInternalService.create(vitamContext, fileFormat);
    }

    @PatchMapping(CommonConstants.PATH_ID)
    public FileFormatDto patch(final @PathVariable("id") String id, @RequestBody final Map<String, Object> partialDto) {
        LOGGER.debug("Patch {} with {}", id, partialDto);
        final VitamContext vitamContext = securityService.buildVitamContext(securityService.getTenantIdentifier());
        Assert.isTrue(
            StringUtils.equals(id, (String) partialDto.get("id")),
            "The DTO identifier must match the path identifier for update."
        );
        return fileFormatInternalService.patch(vitamContext, partialDto);
    }

    @DeleteMapping(CommonConstants.PATH_ID)
    public void delete(final @PathVariable("id") String id) {
        LOGGER.debug("Delete {}", id);
        final VitamContext vitamContext = securityService.buildVitamContext(securityService.getTenantIdentifier());
        fileFormatInternalService.delete(vitamContext, id);
    }

    @GetMapping("/export")
    public ResponseEntity<Resource> export() {
        LOGGER.debug("Export File Formats");
        final VitamContext vitamContext = securityService.buildVitamContext(securityService.getTenantIdentifier());

        Response response = fileFormatInternalService.export(vitamContext);
        Object entity = response.getEntity();
        if (entity instanceof InputStream) {
            Resource resource = new InputStreamResource((InputStream) entity);
            return new ResponseEntity<>(resource, HttpStatus.OK);
        }
        return null;
    }

    private JsonNode findHistoryById(final @PathVariable("id") String id) throws VitamClientException {
        final VitamContext vitamContext = securityService.buildVitamContext(securityService.getTenantIdentifier());
        LOGGER.debug("get logbook for file format with id :{}", id);
        return fileFormatInternalService.findHistoryByIdentifier(vitamContext, id);
    }

    @PostMapping(CommonConstants.PATH_IMPORT)
    @Consumes(MediaType.APPLICATION_OCTET_STREAM_VALUE)
    @Produces(MediaType.APPLICATION_OCTET_STREAM_VALUE)
    public JsonNode importFileFormats(
        @RequestParam("fileName") String fileName,
        @RequestParam("file") MultipartFile file
    ) {
        SafeFileChecker.checkSafeFilePath(file.getOriginalFilename());
        SanityChecker.isValidFileName(fileName);
        SafeFileChecker.checkSafeFilePath(fileName);
        LOGGER.debug("import file format file {}", fileName);
        final VitamContext vitamContext = securityService.buildVitamContext(securityService.getTenantIdentifier());
        return fileFormatInternalService.importFileFormats(vitamContext, fileName, file);
    }
}
