/**
 * Copyright French Prime minister Office/SGMAP/DINSIC/Vitam Program (2019-2020)
 * and the signatories of the "VITAM - Accord du Contributeur" agreement.
 *
 * contact@programmevitam.fr
 *
 * This software is a computer program whose purpose is to implement
 * implement a digital archiving front-office system for the secure and
 * efficient high volumetry VITAM solution.
 *
 * This software is governed by the CeCILL-C license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL-C
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided only
 * with a limited warranty  and the software's author,  the holder of the
 * economic rights,  and the successive licensors  have only  limited
 * liability.
 *
 * In this respect, the user's attention is drawn to the risks associated
 * with loading,  using,  modifying and/or developing or reproducing the
 * software by the user in light of its specific status of free software,
 * that may mean  that it is complicated to manipulate,  and  that  also
 * therefore means  that it is reserved for developers  and  experienced
 * professionals having in-depth computer knowledge. Users are therefore
 * encouraged to load and test the software's suitability as regards their
 * requirements in conditions enabling the security of their systems and/or
 * data to be ensured and,  more generally, to use and operate it in the
 * same conditions as regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL-C license and that you accept its terms.
 */
package fr.gouv.vitamui.iam.internal.server.rest;

import fr.gouv.vitamui.common.security.SanityChecker;
import fr.gouv.vitamui.commons.api.domain.ApplicationDto;
import fr.gouv.vitamui.commons.api.logger.VitamUILogger;
import fr.gouv.vitamui.commons.api.logger.VitamUILoggerFactory;
import fr.gouv.vitamui.commons.api.utils.EnumUtils;
import fr.gouv.vitamui.iam.common.dto.common.EmbeddedOptions;
import fr.gouv.vitamui.iam.common.rest.RestApi;
import fr.gouv.vitamui.iam.internal.server.application.service.ApplicationInternalService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.Getter;
import lombok.Setter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.Collection;
import java.util.Optional;

/**
 * The controller to check list available applications.
 *
 *
 */
@RestController
@RequestMapping(RestApi.V1_APPLICATIONS_URL)
@Getter
@Setter
@Api(tags = "applications", value = "Applications Management", description = "Applications Management")
public class ApplicationInternalController {

    private static final VitamUILogger LOGGER = VitamUILoggerFactory.getInstance(ApplicationInternalController.class);

    private ApplicationInternalService internalApplicationService;

    @Autowired
    public ApplicationInternalController(final ApplicationInternalService internalApplicationService) {
        this.internalApplicationService = internalApplicationService;
    }

    /**
     * Get All applications filtered by user privileges
     * @param criteria
     * @param embedded
     * @return all Applications matching user privileges
     */
    @GetMapping
    public Collection<ApplicationDto> getAll(
        final Optional<String> criteria,
        @RequestParam final Optional<String> embedded
    ) {
        LOGGER.debug("Get all with criteria={}, embedded={}", criteria, embedded);
        SanityChecker.sanitizeCriteria(criteria);
        EnumUtils.checkValidEnum(EmbeddedOptions.class, embedded);
        return internalApplicationService.getAll(criteria, embedded);
    }

    @GetMapping(path = "/{identifier}/externalid")
    @ApiOperation(value = "Check if an application can have an external identifier")
    public boolean isApplicationExternalIdentifierEnabled(final @PathVariable("identifier") String applicationId) {
        return internalApplicationService.isApplicationExternalIdentifierEnabled(applicationId);
    }
}
