/*
 * Decompiled with CFR 0.152.
 */
package fr.gouv.vitam.storage.offers.tape.cas;

import com.google.common.annotations.VisibleForTesting;
import fr.gouv.vitam.common.LocalDateUtil;
import fr.gouv.vitam.common.ParametersChecker;
import fr.gouv.vitam.common.alert.AlertService;
import fr.gouv.vitam.common.alert.AlertServiceImpl;
import fr.gouv.vitam.common.logging.VitamLogger;
import fr.gouv.vitam.common.logging.VitamLoggerFactory;
import fr.gouv.vitam.common.security.IllegalPathException;
import fr.gouv.vitam.common.security.SafeFileChecker;
import fr.gouv.vitam.common.thread.VitamThreadPoolExecutor;
import fr.gouv.vitam.storage.offers.tape.cache.LRUCache;
import fr.gouv.vitam.storage.offers.tape.cache.LRUCacheEntry;
import fr.gouv.vitam.storage.offers.tape.cache.LRUCacheEvictionJudge;
import fr.gouv.vitam.storage.offers.tape.cas.ArchiveCacheEntry;
import fr.gouv.vitam.storage.offers.tape.cas.ArchiveCacheEvictionController;
import fr.gouv.vitam.storage.offers.tape.cas.BucketTopologyHelper;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.time.ZoneOffset;
import java.util.Optional;
import java.util.concurrent.Executor;
import java.util.function.Supplier;
import java.util.stream.Stream;
import javax.annotation.concurrent.ThreadSafe;

@ThreadSafe
public class ArchiveCacheStorage {
    private static final VitamLogger LOGGER = VitamLoggerFactory.getInstance(ArchiveCacheStorage.class);
    private static final long MB_TO_BYTES = 1000000L;
    private final Path cacheDirectory;
    private final BucketTopologyHelper bucketTopologyHelper;
    private final LRUCache<ArchiveCacheEntry> lruCache;
    private final ArchiveCacheEvictionController archiveCacheEvictionController;

    public ArchiveCacheStorage(String cacheDirectory, BucketTopologyHelper bucketTopologyHelper, ArchiveCacheEvictionController archiveCacheEvictionController, long maxStorageSpace, long evictionStorageSpaceThreshold, long safeStorageSpaceThreshold) throws IllegalPathException, IOException {
        this(cacheDirectory, bucketTopologyHelper, archiveCacheEvictionController, maxStorageSpace, evictionStorageSpaceThreshold, safeStorageSpaceThreshold, (Executor)VitamThreadPoolExecutor.getDefaultExecutor(), (AlertService)new AlertServiceImpl());
    }

    @VisibleForTesting
    ArchiveCacheStorage(String cacheDirectory, BucketTopologyHelper bucketTopologyHelper, ArchiveCacheEvictionController archiveCacheEvictionController, long maxStorageSpace, long evictionStorageSpaceThreshold, long safeStorageSpaceThreshold, Executor executor, AlertService alertService) throws IllegalPathException, IOException {
        this.archiveCacheEvictionController = archiveCacheEvictionController;
        this.cacheDirectory = SafeFileChecker.checkSafeDirPath((String)cacheDirectory, (String[])new String[0]).toPath();
        this.bucketTopologyHelper = bucketTopologyHelper;
        this.lruCache = this.createLRUCache(maxStorageSpace, evictionStorageSpaceThreshold, safeStorageSpaceThreshold, this::fileEvictionJudgeFactory, executor, alertService);
    }

    private LRUCacheEvictionJudge<ArchiveCacheEntry> fileEvictionJudgeFactory() {
        LOGGER.warn("Preparing archive cache eviction. Max capacity {}MB, Current usage: {}MB", (Object)(this.lruCache.getMaxCapacity() / 1000000L), (Object)(this.lruCache.getCurrentCapacity() / 1000000L));
        LRUCacheEvictionJudge<ArchiveCacheEntry> entryLRUCacheEvictionJudge = this.archiveCacheEvictionController.computeEvictionJudge();
        LOGGER.info("Done preparing archive cache eviction...");
        return entryLRUCacheEvictionJudge;
    }

    private LRUCache<ArchiveCacheEntry> createLRUCache(long maxCapacity, long evictionCapacity, long safeCapacity, Supplier<LRUCacheEvictionJudge<ArchiveCacheEntry>> archiveEvictionJudgeFactory, Executor evictionExecutor, AlertService alertService) throws IllegalPathException, IOException {
        LRUCache<ArchiveCacheEntry> lRUCache;
        block10: {
            Stream<Path> pathStream = Files.walk(this.cacheDirectory, 2, new FileVisitOption[0]);
            try {
                Stream initialFileCacheEntries = pathStream.filter(this::filterNonRegularFiles).peek(this::checkNonRootFile).peek(this::checkFileSafety).peek(this::checkFileBucketId).map(this::createFileCacheEntry);
                lRUCache = new LRUCache<ArchiveCacheEntry>(maxCapacity, evictionCapacity, safeCapacity, archiveEvictionJudgeFactory, this::evictFileListener, initialFileCacheEntries, evictionExecutor, alertService);
                if (pathStream == null) break block10;
            }
            catch (Throwable initialFileCacheEntries) {
                try {
                    if (pathStream != null) {
                        try {
                            pathStream.close();
                        }
                        catch (Throwable throwable) {
                            initialFileCacheEntries.addSuppressed(throwable);
                        }
                    }
                    throw initialFileCacheEntries;
                }
                catch (RuntimeException e) {
                    Throwable cause = e.getCause();
                    if (cause instanceof IllegalPathException) {
                        throw (IllegalPathException)cause;
                    }
                    if (cause instanceof IOException) {
                        throw (IOException)cause;
                    }
                    throw e;
                }
            }
            pathStream.close();
        }
        return lRUCache;
    }

    public void reserveArchiveStorageSpace(String fileBucketId, String tarId, long fileSize) throws IllegalPathException {
        Path filePath = this.checkSafeDirPath(fileBucketId, tarId);
        ArchiveCacheEntry archiveCacheEntry = new ArchiveCacheEntry(fileBucketId, tarId);
        if (this.lruCache.containsEntry(archiveCacheEntry)) {
            throw new IllegalArgumentException("Archive '" + archiveCacheEntry + "' already exists in cache.");
        }
        if (this.lruCache.isReservedEntry(archiveCacheEntry)) {
            throw new IllegalArgumentException("Reservation for file '" + archiveCacheEntry + "' already exists.");
        }
        if (Files.exists(filePath, new LinkOption[0])) {
            throw new IllegalArgumentException("File '" + archiveCacheEntry + "' already exists on storage directory " + this.cacheDirectory);
        }
        Instant currentTimestamp = this.getCurrentInstant();
        this.lruCache.reserveEntry(new LRUCacheEntry<ArchiveCacheEntry>(archiveCacheEntry, fileSize, currentTimestamp));
    }

    public void moveArchiveToCache(Path initialFilePath, String fileBucketId, String tarId) throws IllegalPathException, IOException {
        Path cachedFilePath = this.checkSafeDirPath(fileBucketId, tarId);
        if (!Files.exists(initialFilePath, new LinkOption[0])) {
            throw new FileNotFoundException("File not exists" + initialFilePath);
        }
        if (!Files.isRegularFile(initialFilePath, new LinkOption[0])) {
            throw new IOException("Not a file" + initialFilePath);
        }
        if (Files.exists(cachedFilePath, new LinkOption[0])) {
            throw new IOException("File already exists" + cachedFilePath);
        }
        ArchiveCacheEntry archiveCacheEntry = new ArchiveCacheEntry(fileBucketId, tarId);
        if (!this.lruCache.isReservedEntry(archiveCacheEntry)) {
            throw new IllegalArgumentException("File " + archiveCacheEntry + " is not reserved in cache");
        }
        LRUCacheEntry<ArchiveCacheEntry> reservedEntry = this.lruCache.getReservedEntry(archiveCacheEntry);
        long fileSize = Files.size(initialFilePath);
        if (reservedEntry.getWeight() != fileSize) {
            throw new IllegalArgumentException("File '" + archiveCacheEntry + "' size " + fileSize + " does not match reserved file capacity " + reservedEntry.getWeight());
        }
        Files.createDirectories(cachedFilePath.getParent(), new FileAttribute[0]);
        Files.move(initialFilePath, cachedFilePath, StandardCopyOption.ATOMIC_MOVE);
        Instant currentTimestamp = this.getCurrentInstant();
        this.lruCache.updateEntryAccessTimestamp(archiveCacheEntry, currentTimestamp);
        this.lruCache.confirmReservation(archiveCacheEntry);
    }

    public void cancelReservedArchive(String fileBucketId, String tarId) {
        ParametersChecker.checkParameter((String)"Missing fileBucketId", (String[])new String[]{fileBucketId});
        ParametersChecker.checkParameter((String)"Missing tarId", (String[])new String[]{tarId});
        ArchiveCacheEntry archiveCacheEntry = new ArchiveCacheEntry(fileBucketId, tarId);
        if (!this.lruCache.isReservedEntry(archiveCacheEntry)) {
            throw new IllegalArgumentException("File " + archiveCacheEntry + " is not reserved in cache");
        }
        this.lruCache.cancelReservation(archiveCacheEntry);
    }

    public Optional<FileInputStream> tryReadArchive(String fileBucketId, String tarId) throws IllegalPathException {
        Path cachedFilePath = this.checkSafeDirPath(fileBucketId, tarId);
        if (!this.containsArchive(fileBucketId, tarId)) {
            return Optional.empty();
        }
        try {
            Optional<FileInputStream> inputStream = Optional.of(new FileInputStream(cachedFilePath.toFile()));
            ArchiveCacheEntry archiveCacheEntry = new ArchiveCacheEntry(fileBucketId, tarId);
            LOGGER.debug("Access to file " + archiveCacheEntry);
            this.lruCache.updateEntryAccessTimestamp(archiveCacheEntry, this.getCurrentInstant());
            return inputStream;
        }
        catch (FileNotFoundException e) {
            LOGGER.warn("Could not open file for read. Concurrent purge?", (Throwable)e);
            return Optional.empty();
        }
    }

    public boolean containsArchive(String fileBucketId, String tarId) {
        ParametersChecker.checkParameter((String)"Missing fileBucketId", (String[])new String[]{fileBucketId});
        ParametersChecker.checkParameter((String)"Missing tarId", (String[])new String[]{tarId});
        ArchiveCacheEntry archiveCacheEntry = new ArchiveCacheEntry(fileBucketId, tarId);
        return this.lruCache.containsEntry(archiveCacheEntry);
    }

    public boolean isArchiveReserved(String fileBucketId, String tarId) {
        ParametersChecker.checkParameter((String)"Missing fileBucketId", (String[])new String[]{fileBucketId});
        ParametersChecker.checkParameter((String)"Missing tarId", (String[])new String[]{tarId});
        ArchiveCacheEntry archiveCacheEntry = new ArchiveCacheEntry(fileBucketId, tarId);
        return this.lruCache.isReservedEntry(archiveCacheEntry);
    }

    public long getMaxStorageSpace() {
        return this.lruCache.getMaxCapacity();
    }

    public long getEvictionStorageSpaceThreshold() {
        return this.lruCache.getEvictionCapacity();
    }

    public long getSafeStorageSpaceThreshold() {
        return this.lruCache.getSafeCapacity();
    }

    public long getCurrentStorageSpaceUsage() {
        return this.lruCache.getCurrentCapacity();
    }

    private Path checkSafeDirPath(String fileBucketId, String tarId) throws IllegalPathException {
        ParametersChecker.checkParameter((String)"Missing fileBucketId", (String[])new String[]{fileBucketId});
        ParametersChecker.checkParameter((String)"Missing tarId", (String[])new String[]{tarId});
        if (!this.bucketTopologyHelper.isValidFileBucketId(fileBucketId)) {
            throw new IllegalArgumentException("Invalid fileBucketId '" + fileBucketId + "'");
        }
        return SafeFileChecker.checkSafeDirPath((String)this.cacheDirectory.toString(), (String[])new String[]{fileBucketId, tarId}).toPath();
    }

    private boolean filterNonRegularFiles(Path filePath) {
        if (!Files.isRegularFile(filePath, new LinkOption[0])) {
            LOGGER.debug("Ignoring non regular file " + filePath);
            return false;
        }
        return true;
    }

    private void checkNonRootFile(Path filePath) {
        Path parentDir = filePath.getParent().toAbsolutePath();
        try {
            if (Files.isSameFile(parentDir, this.cacheDirectory)) {
                throw new IllegalStateException("Invalid file '" + filePath + "'  at root of cache directory " + this.cacheDirectory + ". Expected {fileBucketId}/{tarId} format");
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void checkFileSafety(Path filePath) {
        try {
            String parentDir = filePath.getParent().getFileName().toString();
            String filename = filePath.getFileName().toString();
            Path safeFilePath = SafeFileChecker.checkSafeFilePath((String)this.cacheDirectory.toString(), (String[])new String[]{parentDir, filename}).toPath();
            if (!Files.isSameFile(safeFilePath, filePath)) {
                throw new IllegalPathException("Illegal file '" + filePath + "' path. Expected " + this.cacheDirectory + " folder to be its parent folder");
            }
        }
        catch (IllegalPathException e) {
            throw new RuntimeException(e);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void checkFileBucketId(Path filePath) {
        String fileBucketId = filePath.getParent().getFileName().toString();
        if (!this.bucketTopologyHelper.isValidFileBucketId(fileBucketId)) {
            throw new IllegalStateException("Unknown fileBucketId " + fileBucketId);
        }
    }

    private LRUCacheEntry<ArchiveCacheEntry> createFileCacheEntry(Path filePath) {
        try {
            String fileBucketId = filePath.getParent().getFileName().toString();
            String tarId = filePath.getFileName().toString();
            BasicFileAttributes attr = Files.readAttributes(filePath, BasicFileAttributes.class, new LinkOption[0]);
            Instant lastAccessTime = attr.lastAccessTime().toInstant();
            long size = attr.size();
            return new LRUCacheEntry<ArchiveCacheEntry>(new ArchiveCacheEntry(fileBucketId, tarId), size, lastAccessTime);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void evictFileListener(ArchiveCacheEntry archiveCacheEntry) {
        try {
            LOGGER.info("Deleting unused archive {}/{}", (Object)archiveCacheEntry.getFileBucketId(), (Object)archiveCacheEntry.getTarId());
            Path filePath = this.cacheDirectory.resolve(archiveCacheEntry.getFileBucketId()).resolve(archiveCacheEntry.getTarId());
            Files.delete(filePath);
        }
        catch (IOException e) {
            LOGGER.warn("Could not delete file {}/{}" + archiveCacheEntry.getFileBucketId() + "/" + archiveCacheEntry.getTarId() + " from " + this.cacheDirectory, (Throwable)e);
        }
    }

    public boolean isCacheEvictionRunning() {
        return this.lruCache.isCacheEvictionRunning();
    }

    private Instant getCurrentInstant() {
        return LocalDateUtil.now().toInstant(ZoneOffset.UTC);
    }
}

