/*
 * Decompiled with CFR 0.152.
 */
package fr.gouv.vitam.storage.offers.tape.cas;

import com.google.common.util.concurrent.Uninterruptibles;
import fr.gouv.vitam.common.LocalDateUtil;
import fr.gouv.vitam.common.ParametersChecker;
import fr.gouv.vitam.common.VitamConfiguration;
import fr.gouv.vitam.common.database.server.query.QueryCriteria;
import fr.gouv.vitam.common.database.server.query.QueryCriteriaOperator;
import fr.gouv.vitam.common.exception.InvalidGuidOperationException;
import fr.gouv.vitam.common.guid.GUIDFactory;
import fr.gouv.vitam.common.guid.GUIDReader;
import fr.gouv.vitam.common.logging.VitamLogger;
import fr.gouv.vitam.common.logging.VitamLoggerFactory;
import fr.gouv.vitam.common.model.storage.AccessRequestStatus;
import fr.gouv.vitam.common.thread.VitamThreadFactory;
import fr.gouv.vitam.common.thread.VitamThreadUtils;
import fr.gouv.vitam.storage.engine.common.model.QueueMessageEntity;
import fr.gouv.vitam.storage.engine.common.model.QueueMessageType;
import fr.gouv.vitam.storage.engine.common.model.ReadOrder;
import fr.gouv.vitam.storage.engine.common.model.TapeAccessRequestReferentialEntity;
import fr.gouv.vitam.storage.engine.common.model.TapeArchiveReferentialEntity;
import fr.gouv.vitam.storage.engine.common.model.TapeLibraryOnTapeArchiveStorageLocation;
import fr.gouv.vitam.storage.engine.common.model.TapeLibraryTarObjectStorageLocation;
import fr.gouv.vitam.storage.engine.common.model.TapeObjectReferentialEntity;
import fr.gouv.vitam.storage.engine.common.model.TarEntryDescription;
import fr.gouv.vitam.storage.offers.tape.cas.AccessRequestReferentialRepository;
import fr.gouv.vitam.storage.offers.tape.cas.ArchiveCacheStorage;
import fr.gouv.vitam.storage.offers.tape.cas.ArchiveReferentialRepository;
import fr.gouv.vitam.storage.offers.tape.cas.BucketTopologyHelper;
import fr.gouv.vitam.storage.offers.tape.cas.ObjectReferentialRepository;
import fr.gouv.vitam.storage.offers.tape.exception.AccessRequestReferentialException;
import fr.gouv.vitam.storage.offers.tape.exception.ArchiveReferentialException;
import fr.gouv.vitam.storage.offers.tape.exception.ObjectReferentialException;
import fr.gouv.vitam.storage.offers.tape.exception.QueueException;
import fr.gouv.vitam.storage.offers.tape.spec.QueueRepository;
import fr.gouv.vitam.workspace.api.exception.ContentAddressableStorageBadRequestException;
import fr.gouv.vitam.workspace.api.exception.ContentAddressableStorageException;
import fr.gouv.vitam.workspace.api.exception.ContentAddressableStorageServerException;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.SetUtils;
import org.apache.commons.lang3.RandomUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;

public class AccessRequestManager {
    private static final VitamLogger LOGGER = VitamLoggerFactory.getInstance(AccessRequestManager.class);
    private static final int MAX_ACCESS_REQUEST_SIZE = 100000;
    private static final int MAX_RETRIES = 3;
    private final ObjectReferentialRepository objectReferentialRepository;
    private final ArchiveReferentialRepository archiveReferentialRepository;
    private final AccessRequestReferentialRepository accessRequestReferentialRepository;
    private final ArchiveCacheStorage archiveCacheStorage;
    private final BucketTopologyHelper bucketTopologyHelper;
    private final QueueRepository readWriteQueue;
    private final int maxAccessRequestSize;
    private final int accessRequestExpirationDelay;
    private final TimeUnit accessRequestExpirationUnit;
    private final int accessRequestPurgeDelay;
    private final TimeUnit accessRequestPurgeUnit;
    private final int accessRequestCleanupTaskIntervalDelay;
    private final TimeUnit accessRequestCleanupTaskIntervalUnit;
    private final ScheduledExecutorService cleanupScheduler;

    public AccessRequestManager(ObjectReferentialRepository objectReferentialRepository, ArchiveReferentialRepository archiveReferentialRepository, AccessRequestReferentialRepository accessRequestReferentialRepository, ArchiveCacheStorage archiveCacheStorage, BucketTopologyHelper bucketTopologyHelper, QueueRepository readWriteQueue, int maxAccessRequestSize, int accessRequestExpirationDelay, TimeUnit accessRequestExpirationUnit, int accessRequestPurgeDelay, TimeUnit accessRequestPurgeUnit, int accessRequestCleanupTaskIntervalDelay, TimeUnit accessRequestCleanupTaskIntervalUnit) {
        ParametersChecker.checkParameter((String)"Required parameters", (Object[])new Object[]{objectReferentialRepository, archiveReferentialRepository, accessRequestReferentialRepository, archiveCacheStorage, bucketTopologyHelper});
        ParametersChecker.checkValue((String)"Invalid maxAccessRequestSize", (long)maxAccessRequestSize, (long)1L);
        ParametersChecker.checkValue((String)"Invalid maxAccessRequestSize", (long)100000L, (long)maxAccessRequestSize);
        ParametersChecker.checkValue((String)"Invalid accessRequestExpirationDelay", (long)accessRequestExpirationDelay, (long)1L);
        ParametersChecker.checkParameter((String)"Invalid accessRequestExpirationUnit", (Object[])new Object[]{accessRequestExpirationUnit});
        ParametersChecker.checkValue((String)"Invalid accessRequestPurgeDelay", (long)accessRequestPurgeDelay, (long)1L);
        ParametersChecker.checkParameter((String)"Invalid accessRequestPurgeUnit", (Object[])new Object[]{accessRequestPurgeUnit});
        if (accessRequestPurgeUnit.convert(accessRequestPurgeDelay, TimeUnit.NANOSECONDS) < accessRequestExpirationUnit.convert(accessRequestExpirationDelay, TimeUnit.NANOSECONDS)) {
            throw new IllegalArgumentException("Access request purge cannot occur before access request expiration");
        }
        ParametersChecker.checkValue((String)"Invalid accessRequestCleanupTaskIntervalDelay", (long)accessRequestCleanupTaskIntervalDelay, (long)1L);
        ParametersChecker.checkParameter((String)"Invalid accessRequestCleanupTaskIntervalUnit", (Object[])new Object[]{accessRequestCleanupTaskIntervalUnit});
        this.objectReferentialRepository = objectReferentialRepository;
        this.archiveReferentialRepository = archiveReferentialRepository;
        this.accessRequestReferentialRepository = accessRequestReferentialRepository;
        this.archiveCacheStorage = archiveCacheStorage;
        this.bucketTopologyHelper = bucketTopologyHelper;
        this.readWriteQueue = readWriteQueue;
        this.maxAccessRequestSize = maxAccessRequestSize;
        this.accessRequestExpirationDelay = accessRequestExpirationDelay;
        this.accessRequestExpirationUnit = accessRequestExpirationUnit;
        this.accessRequestPurgeDelay = accessRequestPurgeDelay;
        this.accessRequestPurgeUnit = accessRequestPurgeUnit;
        this.accessRequestCleanupTaskIntervalDelay = accessRequestCleanupTaskIntervalDelay;
        this.accessRequestCleanupTaskIntervalUnit = accessRequestCleanupTaskIntervalUnit;
        this.cleanupScheduler = Executors.newScheduledThreadPool(1, (ThreadFactory)VitamThreadFactory.getInstance());
    }

    public void startExpirationHandler() {
        this.cleanupScheduler.scheduleWithFixedDelay(this::accessRequestCleanupTask, this.accessRequestCleanupTaskIntervalDelay, this.accessRequestCleanupTaskIntervalDelay, this.accessRequestCleanupTaskIntervalUnit);
    }

    public String createAccessRequest(String containerName, List<String> objectNames) throws ContentAddressableStorageException {
        ParametersChecker.checkParameter((String)"Required containerName", (String[])new String[]{containerName});
        ParametersChecker.checkParameter((String)"Required objectNames", (Object[])new Object[]{objectNames});
        ParametersChecker.checkParameter((String)"Required objectNames", (String[])((String[])objectNames.toArray(String[]::new)));
        this.checkRequestSize(objectNames);
        this.checkDuplicateObjectNames(objectNames);
        try {
            List<TapeArchiveReferentialEntity> unavailableArchivesOnDisk = this.getUnavailableArchivesOnDiskForObjects(containerName, objectNames);
            String accessRequestId = AccessRequestManager.generateAccessRequestId();
            List unavailableArchiveIds = unavailableArchivesOnDisk.stream().map(TapeArchiveReferentialEntity::getArchiveId).collect(Collectors.toList());
            LocalDateTime now = LocalDateUtil.now();
            String creationDate = LocalDateUtil.getFormattedDateTimeForMongo((LocalDateTime)now);
            String readyDate = unavailableArchiveIds.isEmpty() ? this.computeReadyDate(now) : null;
            String expirationDate = unavailableArchiveIds.isEmpty() ? this.computeExpirationDate(now) : null;
            String purgeDate = unavailableArchiveIds.isEmpty() ? this.computePurgeDate(now) : null;
            TapeAccessRequestReferentialEntity accessRequest = new TapeAccessRequestReferentialEntity(accessRequestId, containerName, objectNames, creationDate, readyDate, expirationDate, purgeDate, unavailableArchiveIds, VitamThreadUtils.getVitamSession().getTenantId().intValue(), 0);
            this.accessRequestReferentialRepository.insert(accessRequest);
            this.bucketTopologyHelper.getFileBucketFromContainerName(containerName);
            List<ReadOrder> readOrders = this.createReadOrders(containerName, unavailableArchivesOnDisk);
            this.addReadOrdersToQueue(readOrders);
            return accessRequestId;
        }
        catch (AccessRequestReferentialException | ArchiveReferentialException | ObjectReferentialException | QueueException e) {
            throw new ContentAddressableStorageServerException("An error occurred during access request creation.", (Throwable)e);
        }
    }

    public Map<String, AccessRequestStatus> checkAccessRequestStatuses(List<String> accessRequestIds, boolean adminCrossTenantAccessRequestAllowed) throws ContentAddressableStorageException {
        for (String accessRequestId2 : accessRequestIds) {
            AccessRequestManager.checkAccessRequestIdFormat(accessRequestId2);
        }
        try {
            HashSet<String> accessRequestIdSet = new HashSet<String>();
            accessRequestIds.forEach(accessRequestId -> {
                if (!accessRequestIdSet.add((String)accessRequestId)) {
                    throw new IllegalArgumentException("Duplicate access Request Id '" + accessRequestId + "'");
                }
            });
            List<TapeAccessRequestReferentialEntity> accessRequestEntities = this.accessRequestReferentialRepository.findByRequestIds(accessRequestIdSet);
            String now = LocalDateUtil.nowFormatted();
            boolean skipTenantCheck = this.skipTenantCheck(adminCrossTenantAccessRequestAllowed);
            int tenantId = VitamThreadUtils.getVitamSession().getTenantId();
            HashMap<String, AccessRequestStatus> results = new HashMap<String, AccessRequestStatus>();
            for (TapeAccessRequestReferentialEntity accessRequestEntity : accessRequestEntities) {
                if (accessRequestEntity.getTenant() != tenantId && !skipTenantCheck) {
                    LOGGER.warn("Illegal access to AccessRequestId " + accessRequestEntity.getRequestId() + " of tenant " + accessRequestEntity.getTenant() + " from tenant " + tenantId);
                    results.putIfAbsent(accessRequestEntity.getRequestId(), AccessRequestStatus.NOT_FOUND);
                    continue;
                }
                if (accessRequestEntity.getExpirationDate() != null && accessRequestEntity.getExpirationDate().compareTo(now) < 0) {
                    results.put(accessRequestEntity.getRequestId(), AccessRequestStatus.EXPIRED);
                    continue;
                }
                if (CollectionUtils.isEmpty((Collection)accessRequestEntity.getUnavailableArchiveIds())) {
                    results.put(accessRequestEntity.getRequestId(), AccessRequestStatus.READY);
                    continue;
                }
                results.put(accessRequestEntity.getRequestId(), AccessRequestStatus.NOT_READY);
            }
            for (String accessRequestId3 : accessRequestIds) {
                results.putIfAbsent(accessRequestId3, AccessRequestStatus.NOT_FOUND);
            }
            return results;
        }
        catch (AccessRequestReferentialException e) {
            throw new ContentAddressableStorageServerException("An error occurred during access request status check by ids", (Throwable)e);
        }
    }

    private boolean skipTenantCheck(boolean adminCrossTenantAccessRequestAllowed) {
        int tenantId = VitamThreadUtils.getVitamSession().getTenantId();
        return adminCrossTenantAccessRequestAllowed && tenantId == VitamConfiguration.getAdminTenant();
    }

    public void removeAccessRequest(String accessRequestId, boolean adminCrossTenantAccessRequestAllowed) throws ContentAddressableStorageException {
        AccessRequestManager.checkAccessRequestIdFormat(accessRequestId);
        try {
            Optional<TapeAccessRequestReferentialEntity> accessRequestEntity = this.accessRequestReferentialRepository.findByRequestId(accessRequestId);
            if (accessRequestEntity.isEmpty()) {
                LOGGER.warn("No such access request " + accessRequestId + ". Already deleted ?");
                return;
            }
            boolean skipTenantCheck = this.skipTenantCheck(adminCrossTenantAccessRequestAllowed);
            int tenantId = VitamThreadUtils.getVitamSession().getTenantId();
            if (accessRequestEntity.get().getTenant() != tenantId && !skipTenantCheck) {
                LOGGER.warn("Illegal access to AccessRequestId " + accessRequestEntity.get().getRequestId() + " of tenant " + accessRequestEntity.get().getTenant() + " from tenant " + tenantId);
                return;
            }
            boolean deleted = this.accessRequestReferentialRepository.deleteAccessRequestById(accessRequestId);
            if (deleted) {
                this.cancelReadOrder(accessRequestEntity.get());
            }
        }
        catch (AccessRequestReferentialException | ArchiveReferentialException | QueueException e) {
            throw new ContentAddressableStorageServerException("An error occurred during access request delete by id '" + accessRequestId + "'");
        }
    }

    public boolean checkObjectAvailability(String containerName, List<String> objectNames) throws ContentAddressableStorageException {
        ParametersChecker.checkParameter((String)"Required containerName", (String[])new String[]{containerName});
        ParametersChecker.checkParameter((String)"Required objectNames", (Object[])new Object[]{objectNames});
        ParametersChecker.checkParameter((String)"Required objectNames", (String[])((String[])objectNames.toArray(String[]::new)));
        this.checkRequestSize(objectNames);
        this.checkDuplicateObjectNames(objectNames);
        try {
            List<TapeArchiveReferentialEntity> unavailableArchivesOnDisk = this.getUnavailableArchivesOnDiskForObjects(containerName, objectNames);
            if (unavailableArchivesOnDisk.isEmpty()) {
                LOGGER.debug("Immediate access is available for objects {} of container {}", objectNames, (Object)containerName);
                return true;
            }
            LOGGER.warn("One or more objects of container {} are not available for on disk. Object names: {}", (Object)containerName, objectNames);
            return false;
        }
        catch (ArchiveReferentialException | ObjectReferentialException e) {
            throw new ContentAddressableStorageServerException("An error occurred while checking object availability.", (Throwable)e);
        }
    }

    private void checkRequestSize(List<String> objectsNames) throws ContentAddressableStorageBadRequestException {
        if (objectsNames.isEmpty()) {
            throw new ContentAddressableStorageBadRequestException("Empty request");
        }
        if (objectsNames.size() > this.maxAccessRequestSize) {
            throw new ContentAddressableStorageBadRequestException("Request too large. Object count: " + objectsNames.size() + ", max: " + this.maxAccessRequestSize);
        }
    }

    private void checkDuplicateObjectNames(List<String> objectsNames) throws ContentAddressableStorageBadRequestException {
        HashSet uniqueObjectNames = new HashSet();
        Set duplicateNames = objectsNames.stream().filter(objectName -> !uniqueObjectNames.add(objectName)).limit(10L).collect(Collectors.toSet());
        if (!duplicateNames.isEmpty()) {
            throw new ContentAddressableStorageBadRequestException("Invalid request. Duplicate object names " + duplicateNames + " in access request");
        }
    }

    private List<TapeArchiveReferentialEntity> getUnavailableArchivesOnDiskForObjects(String containerName, List<String> objectsIds) throws ObjectReferentialException, ArchiveReferentialException {
        Set<String> tarIds = this.getTarIds(containerName, objectsIds);
        List<TapeArchiveReferentialEntity> onTapeArchives = this.selectArchivesStoredOnTape(tarIds);
        return this.filterArchivesPresentInCache(containerName, onTapeArchives);
    }

    private Set<String> getTarIds(String containerName, List<String> objectNames) throws ObjectReferentialException {
        List<TapeObjectReferentialEntity> objectReferentialEntities = this.objectReferentialRepository.bulkFind(containerName, new HashSet<String>(objectNames));
        return objectReferentialEntities.stream().filter(obj -> obj.getLocation() instanceof TapeLibraryTarObjectStorageLocation).flatMap(obj -> ((TapeLibraryTarObjectStorageLocation)obj.getLocation()).getTarEntries().stream()).map(TarEntryDescription::getTarFileId).collect(Collectors.toSet());
    }

    private List<TapeArchiveReferentialEntity> selectArchivesStoredOnTape(Collection<String> archiveIds) throws ArchiveReferentialException {
        HashSet<String> archiveIdSet = new HashSet<String>(archiveIds);
        List<TapeArchiveReferentialEntity> archiveReferentialEntities = this.archiveReferentialRepository.bulkFind(archiveIdSet);
        if (archiveReferentialEntities.size() != archiveIds.size()) {
            Set foundArchiveIds = archiveReferentialEntities.stream().map(TapeArchiveReferentialEntity::getArchiveId).collect(Collectors.toSet());
            throw new IllegalStateException("Unknown archive ids: " + SetUtils.difference(archiveIdSet, foundArchiveIds));
        }
        return archiveReferentialEntities.stream().filter(tarArchive -> tarArchive.getLocation() instanceof TapeLibraryOnTapeArchiveStorageLocation).collect(Collectors.toList());
    }

    private List<TapeArchiveReferentialEntity> filterArchivesPresentInCache(String containerName, List<TapeArchiveReferentialEntity> onTapeArchives) {
        String fileBucketId = this.bucketTopologyHelper.getFileBucketFromContainerName(containerName);
        return onTapeArchives.stream().filter(archiveEntity -> !this.archiveCacheStorage.containsArchive(fileBucketId, archiveEntity.getArchiveId())).collect(Collectors.toList());
    }

    private List<ReadOrder> createReadOrders(String containerName, List<TapeArchiveReferentialEntity> unavailableArchivesOnDisk) {
        ArrayList<ReadOrder> readOrders = new ArrayList<ReadOrder>();
        String fileBucketId = this.bucketTopologyHelper.getFileBucketFromContainerName(containerName);
        String bucketId = this.bucketTopologyHelper.getBucketFromFileBucket(fileBucketId);
        for (TapeArchiveReferentialEntity archiveEntity : unavailableArchivesOnDisk) {
            TapeLibraryOnTapeArchiveStorageLocation onTapeLocation = (TapeLibraryOnTapeArchiveStorageLocation)archiveEntity.getLocation();
            ReadOrder readOrder = new ReadOrder(onTapeLocation.getTapeCode(), onTapeLocation.getFilePosition(), archiveEntity.getArchiveId(), bucketId, fileBucketId, archiveEntity.getSize().longValue());
            readOrders.add(readOrder);
        }
        return readOrders;
    }

    private void addReadOrdersToQueue(List<ReadOrder> readOrders) throws QueueException {
        for (ReadOrder readOrder : readOrders) {
            this.readWriteQueue.addIfAbsent(Arrays.asList(new QueryCriteria("fileName", (Object)readOrder.getFileName(), QueryCriteriaOperator.EQ), new QueryCriteria("queue_message_type", (Object)QueueMessageType.ReadOrder.name(), QueryCriteriaOperator.EQ)), (QueueMessageEntity)readOrder);
        }
    }

    public void updateAccessRequestWhenArchiveReady(String readyArchiveId) throws AccessRequestReferentialException {
        List<TapeAccessRequestReferentialEntity> accessRequestEntities = this.accessRequestReferentialRepository.findByUnavailableArchiveId(readyArchiveId);
        for (TapeAccessRequestReferentialEntity accessRequestEntity : accessRequestEntities) {
            this.updateAccessRequestWithReadyArchiveId(readyArchiveId, accessRequestEntity);
        }
    }

    private void updateAccessRequestWithReadyArchiveId(String readyArchiveId, TapeAccessRequestReferentialEntity accessRequestEntity) throws AccessRequestReferentialException {
        for (int nbTry = 0; nbTry < 3; ++nbTry) {
            boolean updateSucceeded = this.tryUpdateAccessRequestWithReadyArchiveId(readyArchiveId, accessRequestEntity);
            if (updateSucceeded) {
                return;
            }
            LOGGER.warn("Concurrent update for " + accessRequestEntity.getRequestId() + ". Retry later...");
            Uninterruptibles.sleepUninterruptibly((long)RandomUtils.nextInt((int)10, (int)1000), (TimeUnit)TimeUnit.MILLISECONDS);
            Optional<TapeAccessRequestReferentialEntity> refreshedAccessRequestEntity = this.accessRequestReferentialRepository.findByRequestId(accessRequestEntity.getRequestId());
            if (refreshedAccessRequestEntity.isEmpty()) {
                LOGGER.info("Request id " + accessRequestEntity.getRequestId() + " deleted meanwhile.");
                return;
            }
            accessRequestEntity = refreshedAccessRequestEntity.get();
        }
        throw new AccessRequestReferentialException("Could not update accessRequest " + accessRequestEntity.getRequestId() + ". Aborting after 3 unsuccessful retries");
    }

    private boolean tryUpdateAccessRequestWithReadyArchiveId(String readyArchiveId, TapeAccessRequestReferentialEntity accessRequestEntity) throws AccessRequestReferentialException {
        List updatedUnavailableArchiveIds = accessRequestEntity.getUnavailableArchiveIds().stream().filter(archiveId -> !archiveId.equals(readyArchiveId)).collect(Collectors.toList());
        int updatedVersion = accessRequestEntity.getVersion() + 1;
        String updatedReadyDate = accessRequestEntity.getReadyDate();
        String updatedExpirationDate = accessRequestEntity.getExpirationDate();
        String updatedPurgeDate = accessRequestEntity.getPurgeDate();
        if (accessRequestEntity.getReadyDate() == null && updatedUnavailableArchiveIds.isEmpty()) {
            LocalDateTime now = LocalDateUtil.now();
            updatedReadyDate = this.computeReadyDate(now);
            updatedExpirationDate = this.computeExpirationDate(now);
            updatedPurgeDate = this.computePurgeDate(now);
        }
        TapeAccessRequestReferentialEntity updatedAccessRequestEntity = new TapeAccessRequestReferentialEntity(accessRequestEntity.getRequestId(), accessRequestEntity.getContainerName(), accessRequestEntity.getObjectNames(), accessRequestEntity.getCreationDate(), updatedReadyDate, updatedExpirationDate, updatedPurgeDate, updatedUnavailableArchiveIds, accessRequestEntity.getTenant(), updatedVersion);
        return this.accessRequestReferentialRepository.updateAccessRequest(updatedAccessRequestEntity, accessRequestEntity.getVersion());
    }

    private void accessRequestCleanupTask() {
        String initialThreadName = Thread.currentThread().getName();
        try {
            Thread.currentThread().setName(initialThreadName + "-AccessRequestCleanupThread");
            this.fixAccessRequestReadyStatus();
            this.deleteExpiredAccessRequests();
        }
        catch (Exception e) {
            LOGGER.error("An error occurred during access request cleanup", (Throwable)e);
        }
        finally {
            Thread.currentThread().setName(initialThreadName);
        }
    }

    private void fixAccessRequestReadyStatus() throws AccessRequestReferentialException {
        LOGGER.info("Fixing ready status for access requests");
        List<TapeAccessRequestReferentialEntity> nonReadyAccessRequests = this.accessRequestReferentialRepository.findNonReadyAccessRequests();
        HashSet<ImmutablePair> fileBucketIdArchiveIdPairs = new HashSet<ImmutablePair>();
        for (TapeAccessRequestReferentialEntity accessRequest : nonReadyAccessRequests) {
            String fileBucketId = this.bucketTopologyHelper.getFileBucketFromContainerName(accessRequest.getContainerName());
            for (String archiveId : accessRequest.getUnavailableArchiveIds()) {
                fileBucketIdArchiveIdPairs.add(ImmutablePair.of((Object)fileBucketId, (Object)archiveId));
            }
        }
        List actuallyReadyArchiveIds = fileBucketIdArchiveIdPairs.stream().filter(i -> this.archiveCacheStorage.containsArchive((String)i.getLeft(), (String)i.getRight())).map(Pair::getRight).collect(Collectors.toList());
        for (String actuallyReadyArchiveId : actuallyReadyArchiveIds) {
            LOGGER.warn(" ArchiveId: " + actuallyReadyArchiveId + "is actually ready. Fixing non updated access requests statuses");
            this.updateAccessRequestWhenArchiveReady(actuallyReadyArchiveId);
        }
    }

    private void deleteExpiredAccessRequests() throws AccessRequestReferentialException {
        LOGGER.info("Cleaning-up expired access requests");
        List<TapeAccessRequestReferentialEntity> deletedAccessRequests = this.accessRequestReferentialRepository.cleanupAndGetExpiredAccessRequests();
        for (TapeAccessRequestReferentialEntity deletedAccessRequest : deletedAccessRequests) {
            LOGGER.warn("Expired access request " + deletedAccessRequest.getRequestId() + " deleted");
        }
    }

    private void cancelReadOrder(TapeAccessRequestReferentialEntity deletedAccessRequest) throws QueueException, ArchiveReferentialException, AccessRequestReferentialException {
        HashSet<String> archiveIdsToCheck = new HashSet<String>(deletedAccessRequest.getUnavailableArchiveIds());
        if (archiveIdsToCheck.isEmpty()) {
            return;
        }
        Set<String> archiveIdsToCancel = this.accessRequestReferentialRepository.excludeArchiveIdsStillRequiredByAccessRequests(archiveIdsToCheck);
        for (String archiveId : archiveIdsToCancel) {
            this.readWriteQueue.tryCancelIfNotStarted(Arrays.asList(new QueryCriteria("fileName", (Object)archiveId, QueryCriteriaOperator.EQ), new QueryCriteria("queue_message_type", (Object)QueueMessageType.ReadOrder.name(), QueryCriteriaOperator.EQ)));
        }
        for (String archiveId : archiveIdsToCancel) {
            this.createReadOrderIfConcurrentAccessRequestCreated(archiveId);
        }
    }

    private void createReadOrderIfConcurrentAccessRequestCreated(String archiveId) throws AccessRequestReferentialException, ArchiveReferentialException, QueueException {
        List<TapeAccessRequestReferentialEntity> accessRequests = this.accessRequestReferentialRepository.findByUnavailableArchiveId(archiveId);
        if (accessRequests.isEmpty()) {
            LOGGER.debug("No concurrent access request created.");
            return;
        }
        LOGGER.warn("Concurrent access request created for " + archiveId + ". Re-append read order to queue");
        String containerName = accessRequests.get(0).getContainerName();
        String fileBucketId = this.bucketTopologyHelper.getFileBucketFromContainerName(containerName);
        String bucket = this.bucketTopologyHelper.getBucketFromFileBucket(fileBucketId);
        Optional<TapeArchiveReferentialEntity> tapeArchiveReferentialEntity = this.archiveReferentialRepository.find(archiveId);
        if (tapeArchiveReferentialEntity.isEmpty()) {
            throw new IllegalStateException("Archive id not found: " + archiveId);
        }
        if (!(tapeArchiveReferentialEntity.get().getLocation() instanceof TapeLibraryOnTapeArchiveStorageLocation)) {
            throw new IllegalStateException("Archive " + archiveId + " expected to be on tape");
        }
        TapeLibraryOnTapeArchiveStorageLocation tapeLocation = (TapeLibraryOnTapeArchiveStorageLocation)tapeArchiveReferentialEntity.get().getLocation();
        ReadOrder readOrder = new ReadOrder(tapeLocation.getTapeCode(), tapeLocation.getFilePosition(), archiveId, bucket, fileBucketId, tapeArchiveReferentialEntity.get().getSize().longValue());
        this.addReadOrdersToQueue(List.of(readOrder));
    }

    public void shutdown() {
        this.cleanupScheduler.shutdown();
        try {
            if (!this.cleanupScheduler.awaitTermination(1L, TimeUnit.MINUTES)) {
                throw new IllegalStateException("Could not shutdown access request manager");
            }
        }
        catch (InterruptedException e) {
            LOGGER.error("Thread interrupted", (Throwable)e);
            Thread.currentThread().interrupt();
        }
    }

    private String computeReadyDate(LocalDateTime now) {
        return LocalDateUtil.getFormattedDateTimeForMongo((LocalDateTime)now);
    }

    private String computeExpirationDate(LocalDateTime now) {
        return LocalDateUtil.getFormattedDateTimeForMongo((LocalDateTime)now.plus(this.accessRequestExpirationDelay, this.accessRequestExpirationUnit.toChronoUnit()));
    }

    private String computePurgeDate(LocalDateTime now) {
        return LocalDateUtil.getFormattedDateTimeForMongo((LocalDateTime)now.plus(this.accessRequestPurgeDelay, this.accessRequestPurgeUnit.toChronoUnit()));
    }

    public static String generateAccessRequestId() {
        return GUIDFactory.newGUID().getId();
    }

    private static void checkAccessRequestIdFormat(String accessRequestId) throws IllegalArgumentException {
        if (StringUtils.isEmpty((CharSequence)accessRequestId)) {
            throw new IllegalArgumentException("Invalid accessRequestId '" + accessRequestId + "'. Null or empty.");
        }
        try {
            GUIDReader.getGUID((String)accessRequestId);
        }
        catch (InvalidGuidOperationException e) {
            throw new IllegalArgumentException("Invalid accessRequestId '" + accessRequestId + "'", e);
        }
    }
}

