/*
 * Decompiled with CFR 0.152.
 */
package fr.gouv.vitam.storage.offers.tape.cache;

import fr.gouv.vitam.common.ParametersChecker;
import fr.gouv.vitam.common.alert.AlertService;
import fr.gouv.vitam.common.logging.VitamLogLevel;
import fr.gouv.vitam.common.logging.VitamLogger;
import fr.gouv.vitam.common.logging.VitamLoggerFactory;
import fr.gouv.vitam.storage.offers.tape.cache.LRUCacheEntry;
import fr.gouv.vitam.storage.offers.tape.cache.LRUCacheEvictionJudge;
import fr.gouv.vitam.storage.offers.tape.cache.LRUQueue;
import java.time.Instant;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.stream.Stream;
import javax.annotation.concurrent.ThreadSafe;

@ThreadSafe
public class LRUCache<T> {
    private static final VitamLogger LOGGER = VitamLoggerFactory.getInstance(LRUCache.class);
    private final long maxCapacity;
    private final long evictionCapacity;
    private final long safeCapacity;
    private final Supplier<LRUCacheEvictionJudge<T>> lruCacheEvictionJudgeFactory;
    private final Consumer<T> evictionListener;
    private final Executor evictionExecutor;
    private final AlertService alertService;
    private final LRUQueue<T> lruQueue;
    private final Map<T, LRUCacheEntry<T>> pendingEntryQueue;
    private final Map<T, LRUCacheEntry<T>> reservedEntryMap;
    private final Map<T, Long> cacheEntryWeightMap;
    private final AtomicBoolean isAsyncEvictionRunning = new AtomicBoolean();
    private long currentCapacity = 0L;

    public LRUCache(long maxCapacity, long evictionCapacity, long safeCapacity, Supplier<LRUCacheEvictionJudge<T>> evictionJudgeFactory, Consumer<T> evictionListener, Stream<LRUCacheEntry<T>> initialEntries, Executor evictionExecutor, AlertService alertService) throws IllegalArgumentException {
        ParametersChecker.checkValue((String)"maxCapacity mush be greater than evictionCapacity", (long)maxCapacity, (long)(evictionCapacity + 1L));
        ParametersChecker.checkValue((String)"evictionCapacity must be greater than safeCapacity", (long)evictionCapacity, (long)(safeCapacity + 1L));
        ParametersChecker.checkValue((String)"safeCapacity must be positive", (long)safeCapacity, (long)1L);
        ParametersChecker.checkParameter((String)"Missing evictionJudgeFactory", (Object[])new Object[]{evictionJudgeFactory});
        ParametersChecker.checkParameter((String)"Missing evictionListener", (Object[])new Object[]{evictionListener});
        ParametersChecker.checkParameter((String)"Missing initial entries", (Object[])new Object[]{initialEntries});
        ParametersChecker.checkParameter((String)"Missing evictionExecutor", (Object[])new Object[]{evictionExecutor});
        ParametersChecker.checkParameter((String)"Missing alert service", (Object[])new Object[]{alertService});
        this.maxCapacity = maxCapacity;
        this.evictionCapacity = evictionCapacity;
        this.safeCapacity = safeCapacity;
        this.lruCacheEvictionJudgeFactory = evictionJudgeFactory;
        this.evictionListener = evictionListener;
        this.evictionExecutor = evictionExecutor;
        this.alertService = alertService;
        this.lruQueue = new LRUQueue();
        this.pendingEntryQueue = new LinkedHashMap<T, LRUCacheEntry<T>>();
        this.reservedEntryMap = new HashMap<T, LRUCacheEntry<T>>();
        this.cacheEntryWeightMap = new HashMap<T, Long>();
        this.initializeCache(initialEntries);
    }

    private void initializeCache(Stream<LRUCacheEntry<T>> initialEntries) throws IllegalArgumentException {
        initialEntries.forEach(entry -> {
            ParametersChecker.checkParameter((String)"Null entry", (Object[])new Object[]{entry});
            ParametersChecker.checkParameter((String)"Null entry key", (Object[])new Object[]{entry.getKey()});
            ParametersChecker.checkParameter((String)"Null last access instant", (Object[])new Object[]{entry.getLastAccessInstant()});
            ParametersChecker.checkParameter((String)"Entry weight must be positive", (Object[])new Object[]{entry.getWeight()});
            if (this.lruQueue.contains(entry.getKey())) {
                throw new IllegalArgumentException("Duplicate key " + entry.getKey());
            }
            this.lruQueue.add(entry.getKey(), entry.getLastAccessInstant().toEpochMilli());
            this.cacheEntryWeightMap.put(entry.getKey(), entry.getWeight());
            this.currentCapacity += entry.getWeight();
            LOGGER.debug("Added an entry '{}' with weight {} and last access time: {} during initialization", new Object[]{entry.getKey(), entry.getWeight(), entry.getLastAccessInstant()});
        });
        this.startAsyncEvictionProcessIfNeeded();
    }

    public synchronized void reserveEntry(LRUCacheEntry<T> entry) throws IllegalArgumentException, IllegalStateException {
        ParametersChecker.checkParameter((String)"Null entry", (Object[])new Object[]{entry});
        ParametersChecker.checkParameter((String)"Null entry key", (Object[])new Object[]{entry.getKey()});
        ParametersChecker.checkParameter((String)"Null last access instant", (Object[])new Object[]{entry.getLastAccessInstant()});
        ParametersChecker.checkParameter((String)"Entry weight must be positive", (Object[])new Object[]{entry.getWeight()});
        if (this.lruQueue.contains(entry.getKey()) || this.pendingEntryQueue.containsKey(entry.getKey()) || this.reservedEntryMap.containsKey(entry.getKey())) {
            throw new IllegalArgumentException("Entry '" + entry.getKey() + "' already exists in the cache.");
        }
        if (this.currentCapacity + entry.getWeight() >= this.maxCapacity) {
            String alertMessage = String.format("Cannot add entry '%s'. Cache capacity exceeded. Max capacity: %d. Eviction capacity: %d. Safe capacity: %d. Current capacity: %d. Entry capacity to reserve : %d", entry.getKey(), this.maxCapacity, this.evictionCapacity, this.safeCapacity, this.currentCapacity, entry.getWeight());
            this.alertService.createAlert(VitamLogLevel.ERROR, alertMessage);
            throw new IllegalStateException(alertMessage);
        }
        this.currentCapacity += entry.getWeight();
        this.reservedEntryMap.put(entry.getKey(), entry);
        this.startAsyncEvictionProcessIfNeeded();
    }

    public synchronized void confirmReservation(T entryKey) throws IllegalArgumentException {
        LRUCacheEntry<T> entry = this.reservedEntryMap.remove(entryKey);
        if (entry == null) {
            throw new IllegalArgumentException("Mo active reservation for entry " + entryKey + ". Reservation already confirmed or canceled?");
        }
        if (this.isAsyncEvictionRunning.get()) {
            this.pendingEntryQueue.put(entryKey, entry);
        } else {
            this.lruQueue.add(entryKey, entry.getLastAccessInstant().toEpochMilli());
        }
        this.cacheEntryWeightMap.put(entryKey, entry.getWeight());
    }

    public synchronized void cancelReservation(T entryKey) {
        LRUCacheEntry<T> entry = this.reservedEntryMap.remove(entryKey);
        if (entry == null) {
            throw new IllegalArgumentException("Mo active reservation for entry " + entryKey + ". Reservation already confirmed or canceled?");
        }
        this.currentCapacity -= entry.getWeight();
    }

    public synchronized boolean updateEntryAccessTimestamp(T entryKey, Instant updatedLastAccessInstant) throws IllegalArgumentException {
        ParametersChecker.checkParameter((String)"Missing entry key", (Object[])new Object[]{entryKey});
        ParametersChecker.checkParameter((String)"Null last access instant", (Object[])new Object[]{updatedLastAccessInstant});
        LRUCacheEntry<T> existingReservedEntry = this.reservedEntryMap.get(entryKey);
        if (existingReservedEntry != null) {
            this.reservedEntryMap.put(entryKey, new LRUCacheEntry<T>(entryKey, existingReservedEntry.getWeight(), updatedLastAccessInstant));
            LOGGER.debug("Updated reserved entry '{}' timestamp", entryKey);
            return true;
        }
        LRUCacheEntry<T> existingPendingEntry = this.pendingEntryQueue.get(entryKey);
        if (existingPendingEntry != null) {
            this.pendingEntryQueue.put(entryKey, new LRUCacheEntry<T>(entryKey, existingPendingEntry.getWeight(), updatedLastAccessInstant));
            LOGGER.debug("Updated pending cache entry '{}' access time to {}", entryKey, (Object)updatedLastAccessInstant);
            return true;
        }
        if (this.lruQueue.update(entryKey, updatedLastAccessInstant.toEpochMilli())) {
            LOGGER.debug("Updated entry '{}' timestamp in queue", entryKey, (Object)updatedLastAccessInstant);
            return true;
        }
        LOGGER.debug("Ignoring Entry '{}'. Entry not found in cache. Concurrent purge?", entryKey);
        return false;
    }

    public synchronized boolean containsEntry(T entryKey) throws IllegalArgumentException {
        ParametersChecker.checkParameter((String)"Missing entry key", (Object[])new Object[]{entryKey});
        return this.lruQueue.contains(entryKey) || this.pendingEntryQueue.containsKey(entryKey);
    }

    public synchronized boolean isReservedEntry(T entryKey) throws IllegalArgumentException {
        ParametersChecker.checkParameter((String)"Missing entry key", (Object[])new Object[]{entryKey});
        return this.reservedEntryMap.containsKey(entryKey);
    }

    public synchronized LRUCacheEntry<T> getReservedEntry(T entryKey) throws IllegalArgumentException {
        ParametersChecker.checkParameter((String)"Missing entry key", (Object[])new Object[]{entryKey});
        return this.reservedEntryMap.get(entryKey);
    }

    public long getMaxCapacity() {
        return this.maxCapacity;
    }

    public long getEvictionCapacity() {
        return this.evictionCapacity;
    }

    public long getSafeCapacity() {
        return this.safeCapacity;
    }

    public synchronized long getCurrentCapacity() {
        return this.currentCapacity;
    }

    private void startAsyncEvictionProcessIfNeeded() {
        if (this.currentCapacity >= this.evictionCapacity && !this.isAsyncEvictionRunning.get()) {
            this.isAsyncEvictionRunning.set(true);
            this.evictionExecutor.execute(this::asyncEvictionProcess);
            LOGGER.info("Cache capacity exceeded. Background eviction process started. Max capacity: {}. Eviction capacity: {}. Safe capacity: {}. Current capacity: {}", new Object[]{this.maxCapacity, this.evictionCapacity, this.safeCapacity, this.currentCapacity});
        }
    }

    private void asyncEvictionProcess() throws IllegalStateException {
        String initialThreadName = Thread.currentThread().getName();
        try {
            Thread.currentThread().setName("CacheEvictionProcess-" + initialThreadName);
            LRUCacheEvictionJudge<T> lruCacheEvictionJudge = this.prepareEvictionJudge();
            this.evictOldEntries(lruCacheEvictionJudge);
        }
        catch (RuntimeException e) {
            LOGGER.error((Throwable)e);
            this.alertService.createAlert(VitamLogLevel.ERROR, "Cache eviction process failed", (Throwable)e);
        }
        finally {
            this.finalizeEvictionProcess();
            Thread.currentThread().setName(initialThreadName);
        }
    }

    private LRUCacheEvictionJudge<T> prepareEvictionJudge() {
        LOGGER.debug("Preparing cache eviction judge...");
        LRUCacheEvictionJudge<T> lruCacheEvictionJudge = this.lruCacheEvictionJudgeFactory.get();
        if (lruCacheEvictionJudge == null) {
            throw new IllegalStateException("Null eviction judge");
        }
        LOGGER.debug("Cache eviction judge initialized.");
        return lruCacheEvictionJudge;
    }

    private synchronized void evictOldEntries(LRUCacheEvictionJudge<T> lruCacheEvictionJudge) {
        LOGGER.info("Cache capacity exceeded. Trying to free some disk space. Max capacity: {}. Eviction capacity: {}. Safe capacity: {}. Current capacity: {}", new Object[]{this.maxCapacity, this.evictionCapacity, this.safeCapacity, this.currentCapacity});
        Iterator<T> lruEntryIterator = this.lruQueue.iterator();
        while (lruEntryIterator.hasNext() && this.currentCapacity >= this.safeCapacity) {
            T entryKeyToEvict = lruEntryIterator.next();
            if (!lruCacheEvictionJudge.canEvictEntry(entryKeyToEvict)) {
                LOGGER.info("Entry {} has not been accessed recently, but is non deletable from cache", entryKeyToEvict);
                continue;
            }
            LOGGER.info("Evicting entry: " + entryKeyToEvict);
            this.currentCapacity -= this.cacheEntryWeightMap.get(entryKeyToEvict).longValue();
            lruEntryIterator.remove();
            this.cacheEntryWeightMap.remove(entryKeyToEvict);
            this.evictionListener.accept(entryKeyToEvict);
        }
        if (this.currentCapacity < this.safeCapacity) {
            LOGGER.info("Enough space freed. Max capacity: {}. Eviction capacity: {}. Safe capacity: {}. Current capacity: {}", new Object[]{this.maxCapacity, this.evictionCapacity, this.safeCapacity, this.currentCapacity});
            return;
        }
        String alertMessage = String.format("Critical cache level. Max capacity: %d. Eviction capacity: %d. Safe capacity: %d. Current capacity: %d", this.maxCapacity, this.evictionCapacity, this.safeCapacity, this.currentCapacity);
        LOGGER.warn(alertMessage);
        this.alertService.createAlert(VitamLogLevel.WARN, alertMessage);
    }

    private synchronized void finalizeEvictionProcess() {
        for (LRUCacheEntry<T> lruCacheEntry : this.pendingEntryQueue.values()) {
            this.lruQueue.add(lruCacheEntry.getKey(), lruCacheEntry.getLastAccessInstant().toEpochMilli());
        }
        this.pendingEntryQueue.clear();
        this.isAsyncEvictionRunning.set(false);
    }

    public boolean isCacheEvictionRunning() {
        return this.isAsyncEvictionRunning.get();
    }
}

